<?php

declare(strict_types=1);
// admin/controller/2fa-setup.php (versão estável, com deriva TOTP)
// Fluxo: mostra instruções, permite revelar QR/segredo (?reveal=1 ou ?show=1)
// e ativa 2FA ao validar um código TOTP.
// Requer tabela `users` com: twofa_enabled (tinyint), twofa_secret (text),
// opcional: twofa_recovery_codes, twofa_trust_until.

if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../twofa_log.php';


// (Opcional) sua lib/classe, se existir. Não dependemos delas para verificar TOTP.
if (file_exists(__DIR__ . '/../twofa_lib.php')) {
  require_once __DIR__ . '/../twofa_lib.php';
}
if (class_exists('TOTP') === false && file_exists(__DIR__ . '/../../classes/TOTP.php')) {
  require_once __DIR__ . '/../../classes/TOTP.php';
}

/* =========================================================================
   Fallbacks/Utilitários 100% locais (não colidem com libs)
   ========================================================================= */

// Gera segredo base32 (alfabeto RFC4648 sem "=")
if (!function_exists('__twofa_generate_secret')) {
  function __twofa_generate_secret(int $length = 32): string {
    $alphabet = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567';
    $bytes = random_bytes($length);
    $out = '';
    for ($i = 0; $i < strlen($bytes); $i++) {
      $out .= $alphabet[ord($bytes[$i]) & 31];
    }
    return $out;
  }
}

// Cripto identidade se a sua lib não definir twofa_encrypt/decrypt
if (!function_exists('twofa_encrypt'))  { function twofa_encrypt(string $s): string { return $s; } }
if (!function_exists('twofa_decrypt'))  { function twofa_decrypt(string $s): string { return $s; } }

// Base32 decode simples
if (!function_exists('__b32_decode')) {
  function __b32_decode(string $b32): string {
    $alphabet = array_flip(str_split('ABCDEFGHIJKLMNOPQRSTUVWXYZ234567'));
    $b32 = strtoupper(preg_replace('/[^A-Z2-7]/', '', $b32));
    $buffer = 0; $bits = 0; $out = '';
    for ($i = 0, $len = strlen($b32); $i < $len; $i++) {
      $val = $alphabet[$b32[$i]] ?? null;
      if ($val === null) continue;
      $buffer = ($buffer << 5) | $val;
      $bits   += 5;
      if ($bits >= 8) {
        $bits -= 8;
        $out  .= chr(($buffer >> $bits) & 0xFF);
      }
    }
    return $out;
  }
}

// HOTP (base p/ TOTP)
if (!function_exists('__hotp_code')) {
  function __hotp_code(string $keyBin, int $counter, int $digits = 6): string {
    $binCounter = pack('N2', ($counter >> 32) & 0xFFFFFFFF, $counter & 0xFFFFFFFF);
    $hash   = hash_hmac('sha1', $binCounter, $keyBin, true);
    $offset = ord(substr($hash, -1)) & 0x0F;
    $trunc  = ((ord($hash[$offset]) & 0x7F) << 24)
            | (ord($hash[$offset + 1]) << 16)
            | (ord($hash[$offset + 2]) << 8)
            | (ord($hash[$offset + 3]));
    $code   = $trunc % (10 ** $digits);
    return str_pad((string)$code, $digits, '0', STR_PAD_LEFT);
  }
}

// Verificador TOTP **usando deriva de relógio APP_TOTP_DRIFT_SEC**
if (!function_exists('__totp_verify')) {
  function __totp_verify(string $secretBase32, string $code, int $window = 2, int $period = 30, int $digits = 6): bool {
    if (!preg_match('/^\d{6}$/', $code)) return false;

    $key = __b32_decode($secretBase32);
    if ($key === '') return false;

    // Aplica deriva (config.php): servidor ADIANTADO => valor negativo; ATRASADO => positivo
    $drift = defined('APP_TOTP_DRIFT_SEC') ? (int)APP_TOTP_DRIFT_SEC : 0;
    $now   = time() + $drift;

    $timeStep = (int) floor($now / $period);
    for ($w = -$window; $w <= $window; $w++) {
      $calc = __hotp_code($key, $timeStep + $w, $digits);
      if (hash_equals($calc, $code)) return true;
    }
    return false;
  }
}

/* =========================================================================
   DB + Autorização
   ========================================================================= */

try { $pdo = connect(); }
catch (Throwable $e) {
  http_response_code(500);
  echo "DB error: " . htmlspecialchars($e->getMessage());
  exit;
}

if (empty($_SESSION['signedin']) || empty($_SESSION['user_email'])) {
  header('Location: ./login.php'); exit;
}

$userEmail = $_SESSION['user_email'];

$st = $pdo->prepare("SELECT user_id, user_email, twofa_enabled, twofa_secret FROM users WHERE user_email = :e LIMIT 1");
$st->execute([':e' => $userEmail]);
$user = $st->fetch(PDO::FETCH_ASSOC);
if (!$user) { http_response_code(404); echo "Usuário não encontrado."; exit; }

$userId       = (int)$user['user_id'];
$twofaEnabled = (int)($user['twofa_enabled'] ?? 0);
$secretRaw    = (string)($user['twofa_secret'] ?? '');
$secretDec    = $secretRaw;

// Descriptografa (se necessário)
if ($secretDec) {
  $tmp = twofa_decrypt($secretDec);
  if (!empty($tmp)) $secretDec = $tmp;
}

// Se ainda não houver segredo, gera e salva
if (!$secretDec) {
  $secretDec = __twofa_generate_secret();
  $secretEnc = twofa_encrypt($secretDec);
  $up = $pdo->prepare("UPDATE users SET twofa_secret = :s WHERE user_id = :id");
  $up->execute([':s' => $secretEnc, ':id' => $userId]);
}

// flags para a view
$showQR = ((isset($_GET['reveal']) && $_GET['reveal'] == '1') || (isset($_GET['show']) && $_GET['show'] == '1'));
$msgOk  = '';
$msgErr = '';

/* =========================================================================
   POST: ativar 2FA (validação do código TOTP)
   ========================================================================= */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['activate'])) {
  $code = trim($_POST['code'] ?? '');
  if (!preg_match('/^\d{6}$/', $code)) {
    $msgErr = 'Código deve ter 6 dígitos.';
  } else {
    // janela=3 (±90s), período=30s, com deriva do config.php aplicada em __totp_verify
    if (__totp_verify($secretDec, $code, 3, 30, 6)) {
      $pdo->prepare("UPDATE users SET twofa_enabled = 1 WHERE user_id = :id")
          ->execute([':id' => $userId]);
      twofa_log('setup.ok', $userEmail);
      $twofaEnabled = 1;
      $msgOk = 'Autenticação em duas etapas ativada com sucesso.';
    } else {
      $msgErr = 'Código inválido. Gere um novo no app e tente novamente.';
    }
  }
}

/* =========================================================================
   Passa dados p/ a view
   ========================================================================= */
$__twofa = [
  'email'    => $userEmail,
  'enabled'  => $twofaEnabled,
  'secret'   => $secretDec,        // mostrado apenas se ?reveal=1 / ?show=1
  'show_qr'  => $showQR,
  'ok'       => $msgOk,
  'err'      => $msgErr,
];

require_once __DIR__ . '/../views/2fa.setup.view.php';
