<?php

declare(strict_types=1);
// admin/controller/2fa-verify.php — verificação 2FA robusta (com fallback de segredo)
// - Respeita APP_TOTP_DRIFT_SEC e APP_TOTP_WINDOW (se definidos no config.php)
// - Tenta descriptografar o segredo; se vier vazio, usa o valor bruto como Base32
// - Aceita códigos de recuperação via twofa_consume_recovery (se existir)

if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../functions.php';

// Carrega libs se existirem (não falha se faltarem)
if (file_exists(__DIR__ . '/../twofa_lib.php')) {
  require_once __DIR__ . '/../twofa_lib.php';
require_once __DIR__ . '/../twofa_log.php';

}
if (!class_exists('TOTP') && file_exists(__DIR__ . '/../../classes/TOTP.php')) {
  require_once __DIR__ . '/../../classes/TOTP.php';
}

// Garante fluxo correto
if (empty($_SESSION['2fa_email_pending'])) {
  header('Location: ./login.php'); exit;
}
$email = $_SESSION['2fa_email_pending'];

/* -------------------- Helpers TOTP (fallback puro) -------------------- */
function __b32dec(string $b32): string {
  $map = array_flip(str_split('ABCDEFGHIJKLMNOPQRSTUVWXYZ234567'));
  $b32 = strtoupper(preg_replace('/[^A-Z2-7]/', '', $b32));
  $buf=0; $bits=0; $out='';
  $len=strlen($b32);
  for($i=0;$i<$len;$i++){
    if(!isset($map[$b32[$i]])) continue;
    $buf = ($buf<<5)|$map[$b32[$i]];
    $bits += 5;
    if($bits>=8){ $bits-=8; $out.=chr(($buf>>$bits)&0xFF); }
  }
  return $out;
}
function __hotp(string $keyBin, int $counter, int $digits=6): string {
  $c = pack('N2', ($counter>>32)&0xFFFFFFFF, $counter&0xFFFFFFFF);
  $h = hash_hmac('sha1', $c, $keyBin, true);
  $o = ord(substr($h,-1)) & 0x0F;
  $dt = ((ord($h[$o])&0x7F)<<24) | (ord($h[$o+1])<<16) | (ord($h[$o+2])<<8) | ord($h[$o+3]);
  $num = $dt % 1000000;
  return str_pad((string)$num, 6, '0', STR_PAD_LEFT);
}
function __totp_verify(string $secretB32, string $code, int $window=2, int $period=30, int $driftSec=0): bool {
  if (!preg_match('/^\d{6}$/', $code)) return false;
  $key = __b32dec($secretB32);
  if ($key==='') return false;
  $t0 = (int) floor((time()+$driftSec)/$period);
  for($w=-$window;$w<=$window;$w++){
    if (hash_equals(__hotp($key,$t0+$w,6), $code)) return true;
  }
  return false;
}
function verify_totp_any(string $secretB32, string $code, int $window=2): bool {
  $drift = defined('APP_TOTP_DRIFT_SEC') ? (int)APP_TOTP_DRIFT_SEC : 0;

  if (class_exists('TOTP')) {
    try {
      $totp = new TOTP($secretB32);
      foreach (['verify','verifyCode','check','validate','isValid','checkCode'] as $m) {
        if (method_exists($totp, $m)) {
          if ($totp->$m($code, $window)) return true;
        }
      }
    } catch (\Throwable $e) { /* fallback abaixo */ }
  }

  return __totp_verify($secretB32, $code, $window, 30, $drift);
}
/* --------------------------------------------------------------------- */

// Busca usuário (preferindo sua lib, senão via PDO)
$err = '';
$row = null;

if (function_exists('twofa_get_row_by_email')) {
  try { $row = twofa_get_row_by_email($email); } catch (\Throwable $e) { $row = null; }
}
if (!$row) {
  try {
    $pdo = connect();
    $st = $pdo->prepare("SELECT user_id, user_email, user_name, twofa_enabled, twofa_secret FROM users WHERE user_email = :e LIMIT 1");
    $st->execute([':e'=>$email]);
    $row = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  } catch (\Throwable $e) {
    $err = 'Erro ao consultar usuário para 2FA.';
  }
}
if (!$row) { header('Location: ./login.php'); exit; }

// Se 2FA não estiver ativado ou não houver segredo salvo, segue login
if (empty($row['twofa_enabled']) || empty($row['twofa_secret'])) {
  $_SESSION['signedin']   = true;
  $_SESSION['user_email'] = $email;
  $_SESSION['user_name']  = $row['user_name'] ?? '';
  unset($_SESSION['2fa_email_pending']);
  header('Location: ./home.php'); exit;
}

// 1) tenta descriptografar (quando twofa_secret é cifrado)
$secret_b32 = '';
try {
  if (function_exists('twofa_decrypt')) {
    $secret_b32 = (string) twofa_decrypt($row['twofa_secret']);
  }
} catch (\Throwable $e) {
  $secret_b32 = '';
}

// 2) fallback: se descriptografia voltou vazia mas há dado no banco,
//    usa o valor BRUTO como Base32.
if ($secret_b32 === '' && !empty($row['twofa_secret'])) {
  $secret_b32 = trim((string)$row['twofa_secret']);
}

// 3) valida o formato Base32 rapidamente (A-Z 2-7)
if ($secret_b32 !== '' && !preg_match('/^[A-Z2-7]+$/i', $secret_b32)) {
  // contém caracteres fora do alfabeto base32; considera inválido
  $secret_b32 = '';
}

// Mensagem se ainda ficou inválido
if ($secret_b32 === '') {
  $err = 'Seu segredo 2FA está vazio ou inválido. Refaça a configuração em Segurança → Autenticação em Duas Etapas.';
}

// POST: validar TOTP/recovery
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $err === '') {
  $code     = preg_replace('/\s+/', '', $_POST['code'] ?? '');
  $remember = !empty($_POST['remember']);
  $recovery = trim($_POST['recovery'] ?? '');

  $ok = false;

  if ($code !== '') {
    $window = defined('APP_TOTP_WINDOW') ? (int)APP_TOTP_WINDOW : 2; // ±60s por padrão
    $ok = verify_totp_any($secret_b32, $code, $window);
  } elseif ($recovery !== '' && function_exists('twofa_consume_recovery')) {
    try { $ok = (bool) twofa_consume_recovery($email, $recovery); } catch (\Throwable $e) { $ok = false; }
  }

  if ($ok) {
    $_SESSION['signedin']   = true;
    $_SESSION['user_email'] = $email;
    $_SESSION['user_name']  = $row['user_name'] ?? '';
    unset($_SESSION['2fa_email_pending']);

    if ($remember && function_exists('twofa_cookie_make')) {
      try { twofa_cookie_make($email, $secret_b32);
    twofa_log('verify.trust.set', $email, ['ip'=>$_SERVER['REMOTE_ADDR'] ?? '']); } catch (\Throwable $e) { /* opcional */ }
    }

    header('Location: ./home.php'); exit;
  } else {
    $err = 'Código inválido. Verifique o app autenticador e tente novamente.';
  }
}

// Render da view
require_once __DIR__ . '/../views/header.view.php';
require_once __DIR__ . '/../views/2fa.verify.view.php';
require_once __DIR__ . '/../views/footer.view.php';
