<?php
declare(strict_types=1);

/**
 * Loader de idiomas para o Admin
 * - Padrão: pt-br
 * - Permite alternar via ?adminlang=pt-br|en|es
 * - Persiste em cookie por 60 dias
 */

if (session_status() === PHP_SESSION_NONE) {
    // Não é obrigatório, mas ajuda caso você use algo da sessão nos langs
    @session_start();
}

$allowed = ['pt-br', 'en', 'es'];
$default = 'pt-br';

// 1) Captura do GET (força atualização imediata do cookie)
$lang = isset($_GET['adminlang']) ? strtolower(trim((string)$_GET['adminlang'])) : '';

// Sanitiza e valida
if (!in_array($lang, $allowed, true)) {
    $lang = '';
}

// 2) Se não veio do GET, tenta o cookie
if ($lang === '' && isset($_COOKIE['adminLang'])) {
    $tmp = strtolower(trim((string)$_COOKIE['adminLang']));
    if (in_array($tmp, $allowed, true)) {
        $lang = $tmp;
    }
}

// 3) Fallback para o padrão
if ($lang === '') {
    $lang = $default;
}

// 4) (Re)grava o cookie (60 dias)
// Ajusta flags conforme seu ambiente (HTTPS => secure=true)
$cookie_options = [
    'expires'  => time() + (60 * 60 * 24 * 60),
    'path'     => '/',
    'secure'   => (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off'),
    'httponly' => false,
    'samesite' => 'Lax',
];
setcookie('adminLang', $lang, $cookie_options);

// 5) Carrega o arquivo de idioma certo
switch ($lang) {
    case 'en':
        require_once __DIR__ . '/lang_en.php';
        break;
    case 'es':
        require_once __DIR__ . '/lang_es.php'; // se não tiver, crie/ajuste
        break;
    case 'pt-br':
    default:
        require_once __DIR__ . '/lang_pt-br.php';
        break;
}
