<?php

declare(strict_types=1);
require_once __DIR__ . '/../classes/TOTP.php';

// === utils cripto (AES-256-CBC) ===
function twofa_key()  { return base64_decode(APP_2FA_ENCRYPTION_KEY_B64); }
function twofa_hmac() { return base64_decode(APP_2FA_HMAC_KEY_B64); }

function twofa_encrypt($plaintext) {
  $key = twofa_key();
  $iv = random_bytes(16);
  $ct = openssl_encrypt($plaintext, 'AES-256-CBC', $key, OPENSSL_RAW_DATA, $iv);
  return base64_encode($iv.$ct);
}
function twofa_decrypt($b64) {
  $blob = base64_decode($b64, true);
  if ($blob === false || strlen($blob) < 17) return '';
  $iv = substr($blob, 0, 16);
  $ct = substr($blob, 16);
  $pt = openssl_decrypt($ct, 'AES-256-CBC', twofa_key(), OPENSSL_RAW_DATA, $iv);
  return $pt !== false ? $pt : '';
}

// === DB helpers ===
function twofa_get_row_by_email($email) {
  $st = connect()->prepare("SELECT * FROM users WHERE user_email = :e LIMIT 1");
  $st->execute([":e"=>$email]);
  return $st->fetch(PDO::FETCH_ASSOC);
}
function twofa_set_enabled($email, $enabled) {
  $st = connect()->prepare("UPDATE users SET twofa_enabled = :en WHERE user_email = :e");
  return $st->execute([":en"=>$enabled?1:0, ":e"=>$email]);
}
function twofa_set_secret_and_codes($email, $secret_b32_enc, $codes_json) {
  $st = connect()->prepare("UPDATE users SET twofa_secret = :s, twofa_recovery_codes = :c WHERE user_email = :e");
  return $st->execute([":s"=>$secret_b32_enc, ":c"=>$codes_json, ":e"=>$email]);
}
function twofa_set_trust_until($email, $until) {
  $st = connect()->prepare("UPDATE users SET twofa_trust_until = :u WHERE user_email = :e");
  return $st->execute([":u"=>$until, ":e"=>$email]);
}

// === recovery codes ===
function twofa_generate_recovery_codes($n=10) {
  $out = [];
  for ($i=0;$i<$n;$i++) {
    $raw = bin2hex(random_bytes(5)); // 10 hex chars
    $out[] = strtoupper($raw);
  }
  return $out;
}
function twofa_hash_codes(array $codes) {
  $hashed = [];
  foreach ($codes as $c) {
    $hashed[] = hash('sha256', $c);
  }
  return json_encode($hashed);
}
function twofa_consume_recovery($email, $codePlain) {
  $row = twofa_get_row_by_email($email);
  if (!$row || empty($row['twofa_recovery_codes'])) return false;
  $arr = json_decode($row['twofa_recovery_codes'], true);
  if (!is_array($arr)) return false;
  $h = hash('sha256', strtoupper(trim($codePlain)));
  $idx = array_search($h, $arr, true);
  if ($idx === false) return false;
  // consome
  array_splice($arr, $idx, 1);
  $st = connect()->prepare("UPDATE users SET twofa_recovery_codes = :c WHERE user_email = :e");
  $st->execute([":c"=>json_encode($arr), ":e"=>$email]);
  return true;
}

// === remember device cookie ===
function twofa_cookie_make($email, $secret_b32, $days=APP_2FA_COOKIE_TTL_DAYS) {
  $exp = time() + ($days*86400);
  $finger = hash('sha1', $secret_b32); // se mudar segredo, invalida
  $payload = $email.'|'.$exp.'|'.$finger;
  $sig = base64_encode(hash_hmac('sha256', $payload, twofa_hmac(), true));
  $blob = base64_encode($payload.'|'.$sig);
  setcookie(APP_2FA_COOKIE_NAME, $blob, [
    'expires'=>$exp, 'path'=>'/admin', 'secure'=>true, 'httponly'=>true, 'samesite'=>'Lax'
  ]);
}

function twofa_cookie_check($email, $secret_b32) {
  if (empty($_COOKIE[APP_2FA_COOKIE_NAME])) return false;
  $blob = base64_decode($_COOKIE[APP_2FA_COOKIE_NAME], true);
  if ($blob === false) return false;
  $parts = explode('|', $blob);
  if (count($parts) < 4) return false;
  list($em,$exp,$finger,$sigB64) = $parts;
  if ($em !== $email) return false;
  if ((int)$exp < time()) return false;
  $payload = $em.'|'.$exp.'|'.$finger;
  $calc = base64_encode(hash_hmac('sha256', $payload, twofa_hmac(), true));
  if (!hash_equals($calc, $sigB64)) return false;
  // segredo atual deve bater com o dedo salvo
  $curr = hash('sha1', $secret_b32);
  return hash_equals($curr, $finger);
}
