<?php
declare(strict_types=1);

namespace ND\Coupons\Support {
    /**
     * Coleção de utilitários de string, números e HTML-escape.
     * Compatível com PHP 8.2+ e PSR-12.
     */
    final class Helpers
    {
        private function __construct() {}

        /**
         * Converte qualquer valor em string de forma segura.
         *
         * @param mixed $value Valor a converter.
         * @return string String resultante ('' se nulo).
         */
        public static function toString(mixed $value): string
        {
            return (string)($value ?? '');
        }

        /**
         * Converte valores numéricos para float; não numéricos viram 0.0.
         *
         * @param mixed $value Valor a converter.
         * @return float Número (ou 0.0).
         */
        public static function toNumber(mixed $value): float
        {
            return is_numeric($value) ? (float) $value : 0.0;
        }

        /**
         * Escapa HTML com charset e flags seguros (contra XSS).
         *
         * @param mixed $value Valor a escapar (é convertido para string).
         * @param int $flags Flags de escaping (padrão seguro).
         * @param string $encoding Charset (UTF-8 por padrão).
         * @return string HTML escapado.
         */
        public static function escapeHtml(
            mixed $value,
            int $flags = ENT_QUOTES | ENT_SUBSTITUTE,
            string $encoding = 'UTF-8'
        ): string {
            return htmlspecialchars(self::toString($value), $flags, $encoding);
        }

        /**
         * Gera URL (ou <img>) do Gravatar com parâmetros tipados.
         *
         * @param string $email Email do usuário.
         * @param int $size Tamanho do avatar (em px).
         * @param string $default Tipo de ícone padrão (ex.: 'mp', 'identicon').
         * @param string $rating Classificação (g, pg, r, x).
         * @param bool $asImg Se true, retorna um <img>; senão, a URL.
         * @param array<string,string|int> $attributes Atributos HTML extras do <img>.
         * @return string URL do gravatar ou tag <img>.
         */
        public static function gravatar(
            string $email,
            int $size = 150,
            string $default = 'mp',
            string $rating = 'g',
            bool $asImg = false,
            array $attributes = []
        ): string {
            $normalized = mb_strtolower(trim($email), 'UTF-8');
            $hash = md5($normalized);
            $url = "https://www.gravatar.com/avatar/{$hash}?s={$size}&d={$default}&r={$rating}";

            if (!$asImg) {
                return $url;
            }

            // Monta atributos HTML de forma segura
            $attr = '';
            foreach ($attributes as $key => $val) {
                $k = preg_replace('/[^a-zA-Z0-9_\-:]/', '', (string) $key) ?? '';
                $v = self::escapeHtml($val);
                $attr .= " {$k}=\"{$v}\"";
            }

            return "<img src=\"{$url}\"{$attr} />";
        }
    }
}

namespace {
    /* ============================================================
     * Funções de compatibilidade NO NAMESPACE GLOBAL (recomendado)
     * Mantêm assinaturas usadas no código legado.
     * ============================================================ */

    if (!function_exists('nd_str')) {
        /**
         * @deprecated Use ND\Coupons\Support\Helpers::toString()
         */
        function nd_str(mixed $v): string
        {
            return \ND\Coupons\Support\Helpers::toString($v);
        }
    }

    if (!function_exists('nd_num')) {
        /**
         * @deprecated Use ND\Coupons\Support\Helpers::toNumber()
         */
        function nd_num(mixed $v): float
        {
            return \ND\Coupons\Support\Helpers::toNumber($v);
        }
    }

    if (!function_exists('parseStoreHoursRange')) {
        /**
         * Extrai horário de abertura e fechamento a partir de um texto
         * no formato "Das 08:00 às 20:00" ou "08:00 - 20:00".
         *
         * @param string|null $hours
         * @return array{open:string,close:string}
         */
        function parseStoreHoursRange(?string $hours): array
        {
            $hours = trim((string) $hours);
            if ($hours === '') {
                return ['open' => '', 'close' => ''];
            }
            if (!preg_match('/(\d{1,2}:\d{2}).*?(\d{1,2}:\d{2})/', $hours, $m)) {
                return ['open' => '', 'close' => ''];
            }
            $open  = $m[1];
            $close = $m[2];
            return ['open' => $open, 'close' => $close];
        }
    }

    if (!function_exists('getStoreHoursPart')) {
        /**
         * Retorna apenas a parte "open" ou "close" de um texto de horário.
         *
         * @param string|null $hours
         * @param string $part "open" ou "close"
         */
        function getStoreHoursPart(?string $hours, string $part = 'open'): string
        {
            $parsed = parseStoreHoursRange($hours);
            return $parsed[$part] ?? '';
        }
    }

    if (!function_exists('isStoreOpenNow')) {
        /**
         * Indica se a loja está aberta agora com base no texto de horário.
         * Retorna:
         *   - true  => aberto
         *   - false => fechado
         *   - null  => horário inválido ou não definido
         *
         * @param string|null $hours
         * @return bool|null
         */
        function isStoreOpenNow(?string $hours): ?bool
        {
            $parsed = parseStoreHoursRange($hours);
            $open  = $parsed['open'] ?? '';
            $close = $parsed['close'] ?? '';

            if ($open === '' || $close === '') {
                return null;
            }

            [$oh, $om] = array_map('intval', explode(':', $open));
            [$ch, $cm] = array_map('intval', explode(':', $close));

            $nowH = (int) date('G');
            $nowM = (int) date('i');

            $now   = $nowH * 60 + $nowM;
            $start = $oh * 60 + $om;
            $end   = $ch * 60 + $cm;

            if ($end < $start) {
                // Faixa cruzando meia-noite
                if ($now >= $start || $now < $end) {
                    return true;
                }
            } else {
                if ($now >= $start && $now < $end) {
                    return true;
                }
            }

            return false;
        }
    }


    if (!function_exists('e')) {
        /**
         * @deprecated Use ND\Coupons\Support\Helpers::escapeHtml()
         */
        function e(mixed $v, int $flags = ENT_QUOTES | ENT_SUBSTITUTE, string $encoding = 'UTF-8'): string
        {
            return \ND\Coupons\Support\Helpers::escapeHtml($v, $flags, $encoding);
        }
    }

    if (!function_exists('getGravatar')) {
        /**
         * @deprecated Use ND\Coupons\Support\Helpers::gravatar()
         */
        function getGravatar(
            string $email,
            int $s = 150,
            string $d = 'mp',
            string $r = 'g',
            bool $img = false,
            array $atts = []
        ): string {
            return \ND\Coupons\Support\Helpers::gravatar($email, $s, $d, $r, $img, $atts);
        }
    }
}
