/*! nd-offer-modals.js v32 (robust wiring + absolute endpoint) */
(function () {
  'use strict';

  var NS = 'nd-report';
  var SEL = {
    form:    '#nd-report-form, form.nd-report-form, #report-modal form',
    sendBtn: '#nd-report-send, .nd-report-send, [data-report-submit]'
  };

  function log(){ try{ console.debug.apply(console, ['['+NS+']'].concat([].slice.call(arguments))); }catch(_){} }
  function hasUIkit(){ return !!(window.UIkit && UIkit.notification && UIkit.modal); }
  function notify(msg, status){
    var now = Date.now(); window.__NDToast = window.__NDToast || {msg:'',ts:0};
    if (window.__NDToast.msg === msg && (now - window.__NDToast.ts) < 1200) return;
    window.__NDToast.msg = msg; window.__NDToast.ts = now;
    if (hasUIkit()) UIkit.notification({message: msg, status: status || 'primary'});
    else console.log('['+NS+']', (status||'info'), msg);
  }
  function $(s,c){ return (c||document).querySelector(s); }
  function on(el,ev,cb,opt){ if (el) el.addEventListener(ev,cb,opt||false); }

  function getDealId(){
    // 1) data-deal-id na página
    var el = document.querySelector('[data-deal-id]');
    if (el && el.getAttribute('data-deal-id')){
      var v = parseInt(el.getAttribute('data-deal-id'), 10);
      if (!isNaN(v)) return v;
    }
    // 2) hidden no form
    var h = document.querySelector(SEL.form+' input[name="deal_id"], '+SEL.form+' input[name="id_oferta"]');
    if (h && h.value){ var n = parseInt(h.value,10); if(!isNaN(n)) return n; }
    // 3) querystring
    var m = location.search.match(/[?&]deal_id=(\d+)/);
    return m ? parseInt(m[1],10) : 0;
  }

  function ensureReasonSelect(form){
    var sel =
      form.querySelector('select[name="reason"]') ||
      form.querySelector('select[id*="reason" i]') ||
      form.querySelector('select');

    if (!sel) return null;
    sel.disabled = false;
    if (!sel.name) sel.name = 'reason';

    // se veio vazio, popula de forma padrão
    if (sel.options.length < 2){
      sel.innerHTML = '';
      [
        ['','Selecione...'],
        ['preco_incorreto','Preço incorreto'],
        ['link_quebrado','Link quebrado'],
        ['conteudo_improprio','Conteúdo impróprio'],
        ['outro','Outro']
      ].forEach(function(p){
        var o = document.createElement('option');
        o.value = p[0]; o.textContent = p[1];
        if (!p[0]) o.selected = true;
        sel.appendChild(o);
      });
    }
    return sel;
  }

  function endpointURL(form){
    // prioriza data-endpoint, senão absoluto padrão
    var e = (form && form.getAttribute('data-endpoint') || '').trim();
    return e || '/controllers/report-save.php';
  }

  function buildPayload(form){
    var fd = new FormData(form);
    var sel = ensureReasonSelect(form);
    var reason = sel ? (sel.value || '') : '';
    var reasonLabel = sel && sel.options[sel.selectedIndex] ? sel.options[sel.selectedIndex].text : '';

    if (!reason) return {error: 'Selecione um motivo para continuar.'};

    fd.set('reason', reason);
    fd.set('reason_label', reasonLabel);
    // aliases aceitos pelo backend
    fd.set('motivo', reason);
    fd.set('motivo_label', reasonLabel);

    var det = (fd.get('details') || fd.get('detalhes') || '').toString();
    fd.set('details', det);
    fd.set('detalhes', det);

    var did = getDealId();
    if (did){ fd.set('deal_id', did); fd.set('id_oferta', did); }

    fd.set('page_url', location.href);
    fd.set('url', location.href);
    fd.set('ajax', '1');

    return {fd: fd};
  }

  function send(form){
    if (!form) { form = $(SEL.form); }
    if (!form) { notify('Formulário de reporte não encontrado.','danger'); return; }
    if (form.__sending) return;

    var payload = buildPayload(form);
    if (payload.error){
      notify(payload.error,'warning');
      return;
    }

    var fd  = payload.fd;
    var url = endpointURL(form);

    form.__sending = true;
    log('POST →', url);

    fetch(url, {
      method: 'POST',
      body: fd,
      headers: {
        'Accept':'application/json',
        'X-Requested-With':'XMLHttpRequest',
        'Cache-Control':'no-store'
      },
      credentials: 'same-origin',
      cache: 'no-store',
      redirect: 'follow'
    })
    .then(function(r){
      if (!r.ok) {
        return r.text().then(function(t){
          throw new Error('HTTP '+r.status+' '+r.statusText+(t?(' — '+t.slice(0,300)):''));
        });
      }
      // tenta json; se falhar, tenta parsear (quando há BOM/echo extras)
      return r.json().catch(function(){
        return r.text().then(function(t){
          try { return JSON.parse(t); }
          catch(_) { throw new Error('Resposta inválida do servidor'); }
        });
      });
    })
    .then(function(ans){
      if (ans && ans.ok){
        notify('Obrigado! Recebemos seu reporte.','success');
        try { form.reset(); } catch(_){}
        try { if (window.UIkit && UIkit.modal) UIkit.modal('#report-modal').hide(); } catch(_){}
      } else {
        throw new Error((ans && ans.error) ? ans.error : 'Falha ao salvar');
      }
    })
    .catch(function(err){
      console.error('['+NS+'] error', err && err.message ? err.message : err);
      notify('Falha: '+(err && err.message ? err.message : 'Tente novamente.'),'danger');
    })
    .finally(function(){
      form.__sending = false;
    });
  }

  /* ---------- WIRING ROBUSTO ---------- */

  // 1) Delegação global de click (pega botões criados depois)
  on(document, 'click', function(ev){
    var target = ev.target && ev.target.closest(SEL.sendBtn);
    if (!target) return;
    var form = target.closest('form') || $(SEL.form);
    ev.preventDefault(); ev.stopPropagation();
    send(form);
  }, {passive:false});

  // 2) Delegação de submit (se o botão for type=submit)
  on(document, 'submit', function(ev){
    var form = ev.target && ev.target.matches(SEL.form) ? ev.target : null;
    if (!form) return;
    ev.preventDefault(); ev.stopPropagation();
    send(form);
  }, {passive:false});

  // 3) Wire imediato se já existir no DOM
  function wireOnce(){
    var form = $(SEL.form);
    if (form && !form.__wired){
      form.__wired = true;
      ensureReasonSelect(form);
      log('form wired', form);
    }
  }

  // 4) Observa entradas no DOM (modal injetado depois)
  var mo;
  try{
    mo = new MutationObserver(function(){
      wireOnce();
    });
    mo.observe(document.documentElement, { childList:true, subtree:true });
  }catch(_){}

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', wireOnce);
  } else {
    wireOnce();
  }

  log('loader ready');
})();
