
/*! ND Reviews v19 – robust render + safe fallback (2025-10-31) */
(function(){
  'use strict';

  var DEBUG = false;
  function log(){ if (DEBUG && window.console) console.log.apply(console, arguments); }

  // --- Helpers ---
  var d = document;
  function qs(s, ctx){ return (ctx||d).querySelector(s); }
  function qsa(s, ctx){ return Array.from((ctx||d).querySelectorAll(s)); }
  function getParam(name){
    var m = (location.search||'').match(new RegExp('[?&]'+name+'=([^&]+)'));
    return m ? decodeURIComponent(m[1]) : '';
  }
  function drawStarsSVG(avgOrInt){
    var v = +avgOrInt || 0;
    v = Math.max(0, Math.min(5, v));
    // full/empty only, arredondado
    var full = Math.round(v);
    var out = '';
    for (var i=1;i<=5;i++){
      var filled = i <= full;
      out += '<svg aria-hidden="true" viewBox="0 0 20 20" class="nd-star-svg" '
           + (filled ? 'data-full="1"' : 'data-full="0"')
           + '><path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.922-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.196-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" /></svg>';
    }
    return '<span class="nd-stars-svg" aria-label="Avaliação: '+full+' de 5">'+out+'</span>';
  }
  function fmtDecimal(n){
    try { return (new Intl.NumberFormat('pt-BR', { minimumFractionDigits:1, maximumFractionDigits:1 })).format(+n||0); }
    catch(e){ return String((+n||0).toFixed(1)).replace('.', ','); }
  }
  function textEsc(s){ return String(s||'').replace(/[&<>]/g, function(c){ return ({'&':'&amp;','<':'&lt;','>':'&gt;'}[c]); }); }
  function normalizeItemId(){
    var id = (window.ND_RATING && window.ND_RATING.item) || getParam('deal_id') || getParam('item') || '';
    return String(id||'').trim();
  }

  // --- DOM targets ---
  var containerList = qs('#nd-reviews');            // lista de reviews
  var containerBox  = qs('#reviews-item');          // bloco pai (tem header e botão)
  var snippet       = qs('.nd-rating-snippet');     // caixa ao lado do preço
  var legacyRow     = containerBox ? qs('.nd-aggregate, .nd-stars', containerBox) : null; // antigas

  // Não esconda nada antes do render!
  function hideLegacyAfterSuccess(){
    if (!containerBox) return;
    var legacy = qsa('.nd-aggregate, .nd-stars', containerBox);
    legacy.forEach(function(el){ el.style.display = 'none'; });
  }

  // Se não existir containerList, cria
  if (!containerList && containerBox){
    containerList = d.createElement('div');
    containerList.id = 'nd-reviews';
    containerBox.appendChild(containerList);
  }

  // --- Fetch data ---
  function endpoint(item){
    var p = new URLSearchParams(); p.set('item', item);
    return '/controllers/get-reviews.php?' + String(p);
  }

  function renderSnippet(stats){
    if (!snippet) return;
    var avg = +(stats && stats.avg) || 0;
    var cnt = +(stats && stats.count) || 0;
    // Substitui qualquer estrelas anteriores por SVG
    var starsHost = snippet.querySelector('.nd-stars') || snippet;
    starsHost.innerHTML = drawStarsSVG(avg);
    // Atualiza nota/contador se existirem ou cria
    var score = snippet.querySelector('.nd-score');
    if (!score){
      score = d.createElement('span'); score.className='nd-score uk-margin-small-left';
      snippet.appendChild(score);
    }
    score.textContent = (avg ? fmtDecimal(avg) : '0,0')+'/5';

    var count = snippet.querySelector('.nd-count');
    if (!count){
      count = d.createElement('span'); count.className='nd-count uk-text-meta uk-margin-small-left';
      snippet.appendChild(count);
    }
    count.textContent = '('+cnt+' avaliação'+(cnt===1?'':'es')+')';
  }

  function avatarHTML(item){
    var url = (item && item.user_avatar) || '';
    var name = String(item && item.user_name || '').trim() || 'Usuário';
    if (url){
      return '<div class="nd-avatar"><img src="'+textEsc(url)+'" alt="'+textEsc(name)+'"></div>';
    }
    var ini = (name.charAt(0)||'U').toUpperCase();
    return '<div class="nd-avatar nd-avatar-fallback" aria-hidden="true">'+ini+'</div>';
  }

  function reviewItemHTML(item){
    var name = String(item.user_name || 'Usuário').trim();
    var when = String(item.created || '').slice(0,10);
    var rating = +item.rating || 0;
    var comment = textEsc(item.comment || '');
    var long = comment.length > 220;
    var body = long ? ('<span class="nd-cmt-short">'+comment.slice(0,220)+'</span><span class="nd-cmt-ellipsis">… </span><a href="#" class="nd-cmt-more">ver mais</a><span class="nd-cmt-full" hidden>'+comment+'</span>')
                    : comment;
    return ''
      + '<article class="nd-review">'
      +   avatarHTML(item)
      +   '<div class="nd-review-body">'
      +     '<header class="nd-review-head">'
      +       '<div class="nd-review-meta"><span class="nd-review-name">'+textEsc(name)+'</span>'
      +       (when? ' <span class="nd-review-date uk-text-meta">'+textEsc(when)+'</span>':'')
      +       '</div>'
      +       '<div class="nd-review-stars">'+drawStarsSVG(rating)+' <span class="nd-review-note">'+(rating||0)+'/5</span></div>'
      +     '</header>'
      +     (comment ? '<p class="nd-review-text">'+body+'</p>' : '')
      +   '</div>'
      + '</article>';
  }

  function bindReadMore(host){
    host.addEventListener('click', function(e){
      var a = e.target.closest('.nd-cmt-more'); if(!a) return;
      e.preventDefault();
      var wrap = a.closest('.nd-review-text'); if(!wrap) return;
      var short = wrap.querySelector('.nd-cmt-short');
      var ell   = wrap.querySelector('.nd-cmt-ellipsis');
      var full  = wrap.querySelector('.nd-cmt-full');
      if (!short || !ell || !full) return;
      if (full.hasAttribute('hidden')){
        short.hidden = true; ell.hidden = true; full.hidden = false;
        a.textContent = 'ver menos';
      }else{
        short.hidden = false; ell.hidden = false; full.hidden = true;
        a.textContent = 'ver mais';
      }
    });
  }

  function renderList(json){
    if (!containerList || !json) return;
    var items = Array.isArray(json.items) ? json.items : [];
    var html = items.map(reviewItemHTML).join('') || '<div class="uk-text-meta">Seja o primeiro a avaliar.</div>';
    containerList.innerHTML = '<div class="nd-reviews-list">'+html+'</div>';
    bindReadMore(containerList);
  }

  function renderHeaderCount(stats){
    if (!containerBox) return;
    var h = qs('.section_title_details', containerBox) || qs('.reviews .section_title_details', containerBox);
    if (!h) return;
    var cnt = +(stats && stats.count) || 0;
    h.textContent = ' ('+cnt+')';
  }

  function updateAll(json){
    try{
      var stats = json && json.stats ? json.stats : {avg:0, count:0};
      renderSnippet(stats);
      renderHeaderCount(stats);
      renderList(json);
      hideLegacyAfterSuccess();
    }catch(e){ log('render error', e); }
  }

  function start(){
    var item = normalizeItemId();
    if (!item){ log('no item'); return; }
    // Carrega CSS se não estiver presente
    if (!qs('link[href*="nd-reviews-ui.css"]')){
      var lk = d.createElement('link');
      lk.rel = 'stylesheet'; lk.href = '/assets/css/nd-reviews-ui.css?v=19';
      d.head.appendChild(lk);
    }
    fetch(endpoint(item), { credentials:'same-origin' })
      .then(function(r){ return r.json(); })
      .then(function(j){ updateAll(j||{}); })
      .catch(function(err){ log('fetch reviews failed', err); /* NÃO esconde legado em erro */ });
  }

  if (d.readyState === 'loading') d.addEventListener('DOMContentLoaded', start);
  else start();
})();
