<?php
declare(strict_types=1);

/**
 * Cadastro público de associados
 * - Exibe um formulário para criação de novo associado
 * - Grava os dados na tabela `associates`
 */

require_once __DIR__ . '/core.php';            // carrega helpers, $connect, $translation, etc.
require_once __DIR__ . '/admin/functions.php'; // cleardata(), connect(), etc.

// Garante sessão (para eventual uso futuro, rate limiting, etc.)
if (session_status() !== PHP_SESSION_ACTIVE) {
    @session_start();
}

$errors  = array();
$success = '';

$name   = '';
$email  = '';
$phone  = '';
$city   = '';
$state  = '';
$group  = '';
$code   = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // Sanitiza dados com AntiXSS (cleardata)
    $name  = cleardata($_POST['associate_name']  ?? '');
    $email = cleardata($_POST['associate_email'] ?? '');
    $phone = cleardata($_POST['associate_phone'] ?? '');
    $city  = cleardata($_POST['associate_city']  ?? '');
    $state = cleardata($_POST['associate_state'] ?? '');
    $group = cleardata($_POST['associate_group'] ?? '');
    $code  = cleardata($_POST['associate_code']  ?? '');

    $status = 0; // cadastra como inativo; altere para 1 se quiser aprovação automática

    // Validações básicas
    if ($name === '') {
        $errors[] = 'Informe seu nome.';
    }

    if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Informe um e-mail válido.';
    }

    // Valida UF se preenchido
    if ($state !== '' && strlen($state) !== 2) {
        $errors[] = 'Informe a UF com 2 letras (ex: SC).';
    }

    // Verifica se e-mail já está em uso (opcional, apenas alerta)
    if ($email !== '') {
        try {
            $db = connect();
            $checkEmail = $db->prepare('SELECT 1 FROM associates WHERE associate_email = :email LIMIT 1');
            $checkEmail->execute([':email' => $email]);
            if ($checkEmail->fetch()) {
                $errors[] = 'Este e-mail já está cadastrado como associado.';
            }
        } catch (Throwable $e) {
            // Se der erro, apenas ignora esta verificação
        }
    }

    // Se o código vier em branco, gera um automaticamente
    if ($code === '') {
        try {
            $db       = connect();
            $unique   = false;
            $attempts = 0;

            while (!$unique && $attempts < 10) {
                $attempts++;
                $generated = 'A' . strtoupper(substr(sha1(uniqid((string)mt_rand(), true)), 0, 7));
                $check = $db->prepare('SELECT 1 FROM associates WHERE associate_code = :code LIMIT 1');
                $check->execute([':code' => $generated]);
                if (!$check->fetch()) {
                    $unique = true;
                    $code   = $generated;
                }
            }

            if ($code === '') {
                // fallback simples se algo der errado
                $code = 'A' . strtoupper(substr(sha1((string)time(), true), 0, 7));
            }
        } catch (Throwable $e) {
            // fallback em caso de erro
            $code = 'A' . strtoupper(substr(sha1((string)time(), true), 0, 7));
        }
    }

    if (empty($errors)) {
        try {
            $db = connect();

            // Verifica se a coluna associate_group existe na tabela (compatibilidade)
            $hasGroup = false;
            try {
                $checkCol = $db->query("SHOW COLUMNS FROM associates LIKE 'associate_group'");
                if ($checkCol && $checkCol->rowCount() > 0) {
                    $hasGroup = true;
                }
            } catch (Throwable $e) {
                $hasGroup = false;
            }

            if ($hasGroup) {
                $stmt = $db->prepare(
                    'INSERT INTO associates
                    (associate_name, associate_email, associate_phone, associate_city,
                     associate_state, associate_group, associate_code, associate_status, associate_created)
                     VALUES
                    (:name, :email, :phone, :city, :state, :grupo, :code, :status, NOW())'
                );
                $stmt->execute([
                    ':name'   => $name,
                    ':email'  => $email,
                    ':phone'  => $phone,
                    ':city'   => $city,
                    ':state'  => $state,
                    ':grupo'  => $group,
                    ':code'   => $code,
                    ':status' => $status,
                ]);
            } else {
                // fallback se a coluna associate_group não existir
                $stmt = $db->prepare(
                    'INSERT INTO associates
                    (associate_name, associate_email, associate_phone, associate_city,
                     associate_state, associate_code, associate_status, associate_created)
                     VALUES
                    (:name, :email, :phone, :city, :state, :code, :status, NOW())'
                );
                $stmt->execute([
                    ':name'   => $name,
                    ':email'  => $email,
                    ':phone'  => $phone,
                    ':city'   => $city,
                    ':state'  => $state,
                    ':code'   => $code,
                    ':status' => $status,
                ]);
            }

            $success = 'Cadastro realizado com sucesso! Guarde seu código de associado: ' . $code;

            // Limpa campos do formulário (mantém e-mail para conferência)
            $name  = '';
            $phone = '';
            $city  = '';
            $state = '';
            $group = '';
            $code  = $code; // mantém código gerado para exibição

        } catch (Throwable $e) {
            $errors[] = 'Ocorreu um erro ao salvar seu cadastro. Tente novamente em instantes.';
        }
    }
}

// Renderiza usando o header/footer padrão do site
require __DIR__ . '/header.php';
require __DIR__ . '/views/associate-register.view.php';
require __DIR__ . '/footer.php';
