#!/usr/bin/env php
<?php
// cli/geocode_locations.php (versão cURL com UA e cache)
// Uso:
//   php cli/geocode_locations.php --country=BR --rate=1.0 --max=500 --force

declare(strict_types=1);
$root = dirname(__DIR__);
require_once $root . '/core.php';

function curl_get_json(string $url): array {
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_CONNECTTIMEOUT => 15,
    CURLOPT_TIMEOUT => 60,
    CURLOPT_HTTPHEADER => [
      'User-Agent: ND-Cupons-Geocode/1.0 (contato: admin@seu-dominio.com.br)'
    ]
  ]);
  $out = curl_exec($ch);
  $err = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  if ($out === false || $code >= 400) {
    fwrite(STDERR, "Nominatim falhou ($code): $err\n");
    return [];
  }
  $j = json_decode($out, true);
  return is_array($j) ? $j : [];
}

$options = getopt('', ['country::','rate::','max::','force::']);
$country = $options['country'] ?? 'BR';
$rate    = isset($options['rate']) ? (float)$options['rate'] : 1.0;
$max     = isset($options['max'])  ? (int)$options['max']  : 500;
$force   = array_key_exists('force', $options);

$cacheFile = $root . '/storage/geocode_cache.json';
$cache = file_exists($cacheFile) ? (json_decode(file_get_contents($cacheFile), true) ?: []) : [];
$saveCache = function() use ($cacheFile, &$cache) {
  file_put_contents($cacheFile, json_encode($cache, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
};

$pdo = $connect;
$sql = "SELECT location_id, location_title, location_lat, location_lng FROM locations ORDER BY location_id ASC";
$rows = $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);

$upd = $pdo->prepare("UPDATE locations SET location_lat=:lat, location_lng=:lng WHERE location_id=:id");
$imported = 0;

foreach ($rows as $r) {
  $id = (int)$r['location_id'];
  $title = trim((string)$r['location_title']);
  if (!$force && $r['location_lat'] && $r['location_lng']) continue;

  // se o título tiver " - UF" no final, melhor pro match
  $q = preg_match('/[A-Z]{2}$/', $title) ? $title : $title . ', ' . $country;

  // cache
  if (isset($cache[$q])) {
    [$lat,$lng] = $cache[$q];
    $upd->execute([':lat'=>$lat, ':lng'=>$lng, ':id'=>$id]);
    echo "[$id] $title -> CACHE OK ($lat, $lng)\n";
    $imported++;
    if ($imported >= $max) break;
    continue;
  }

  // nominatim
  $url = 'https://nominatim.openstreetmap.org/search?format=json&limit=1&countrycodes='
       . urlencode(strtolower($country)) . '&q=' . urlencode($q);
  echo "[$id] $title -> NOMINATIM: $url\n";
  $arr = curl_get_json($url);
  if ($arr && isset($arr[0]['lat'], $arr[0]['lon'])) {
    $lat = (float)$arr[0]['lat'];
    $lng = (float)$arr[0]['lon'];
    $upd->execute([':lat'=>$lat, ':lng'=>$lng, ':id'=>$id]);
    $cache[$q] = [$lat,$lng];
    $saveCache();
    echo "[$id] $title -> NOMINATIM OK ($lat, $lng)\n";
    $imported++;
  }
  if ($imported >= $max) break;
  usleep((int)($rate * 1_000_000));
}

echo "Atualizados: $imported locations.\n";
$saveCache();
