<?php
declare(strict_types=1);

/**
 * Pequeno bootstrap de conexão e logging.
 * - Tenta ENV (DB_HOST, DB_NAME, DB_USER, DB_PASS) e depois config.php da raiz (array $database).
 * - Exponibiliza nd_db(): PDO singleton
 * - Exponibiliza nd_log(): grava em storage/logs/report-save.log
 */

if (!function_exists('nd_log')) {
    function nd_log(string $line): void {
        $dir = dirname(__DIR__) . '/../storage/logs'; // raiz/storage/logs
        if (!is_dir($dir)) @mkdir($dir, 0775, true);
        $file = $dir . '/report-save.log';
        @file_put_contents($file, '[' . date('Y-m-d H:i:s') . '] ' . $line . PHP_EOL, FILE_APPEND);
    }
}

if (!function_exists('nd_env')) {
    function nd_env(string $key, ?string $default = null): ?string {
        $v = getenv($key);
        if ($v === false && isset($_ENV[$key])) $v = $_ENV[$key];
        return ($v === false) ? $default : $v;
    }
}

if (!function_exists('nd_pdo_from_config')) {
    function nd_pdo_from_config(array $cfg): PDO {
        $host = $cfg['host'] ?? 'localhost';
        $db   = $cfg['db']   ?? ($cfg['name'] ?? '');
        $user = $cfg['user'] ?? ($cfg['username'] ?? '');
        $pass = $cfg['pass'] ?? ($cfg['password'] ?? '');

        if ($host === '' || $db === '' || $user === '') {
            nd_log('Credenciais inválidas em nd_pdo_from_config');
            throw new RuntimeException('Credenciais DB incompletas');
        }

        $dsn = "mysql:host={$host};dbname={$db};charset=utf8mb4";
        $opt = [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
        ];
        return new PDO($dsn, $user, $pass, $opt);
    }
}

if (!function_exists('nd_db')) {
    function nd_db(): PDO {
        static $pdo = null;
        if ($pdo instanceof PDO) return $pdo;

        // 1) ENV
        $h = nd_env('DB_HOST');
        $n = nd_env('DB_NAME');
        $u = nd_env('DB_USER');
        $p = nd_env('DB_PASS');
        if ($h && $n && $u) {
            try {
                $pdo = nd_pdo_from_config(['host'=>$h,'db'=>$n,'user'=>$u,'pass'=>$p ?? '']);
                nd_log("Conexão via ENV OK ({$h}/{$n})");
                return $pdo;
            } catch (Throwable $e) {
                nd_log('Falha ENV: ' . $e->getMessage());
            }
        }

        // 2) config.php na raiz do site
        $configFile = dirname(__DIR__, 2) . '/config.php'; // controllers/lib -> raiz
        if (is_file($configFile)) {
            try {
                $database = null;
                require $configFile; // espera $database = ['host','db','user','pass']
                if (is_array($database)) {
                    $pdo = nd_pdo_from_config($database);
                    nd_log("Conexão via config.php OK ({$database['host']}/{$database['db']})");
                    return $pdo;
                }
                nd_log('config.php não expôs $database');
            } catch (Throwable $e) {
                nd_log('Falha config.php: ' . $e->getMessage());
            }
        } else {
            nd_log('config.php não encontrado em ' . $configFile);
        }

        throw new RuntimeException('Sem conexão com o banco');
    }
}
