<?php

declare(strict_types=1);

// ND - Endpoint leve para registrar evento de "compra"/"pedido"
// relacionado a uma oferta e notificar o comerciante via WhatsApp (opcional).
//
// Pensado para ser não-invasivo:
// - Se WhatsApp não estiver configurado, o envio é ignorado.
// - Se ocorre erro de BD ou de conexão com API, o endpoint responde mesmo assim.
//
// Payload esperado (JSON ou POST):
//   deal_id (obrigatório)
//   purchase_ref (obrigatório)
//   customer_id (opcional)
//   customer_name (opcional)
//   amount (opcional)
//   currency (opcional)

require_once __DIR__ . '/../core.php';

header('Content-Type: application/json; charset=UTF-8');

$payload = [];

$raw = file_get_contents('php://input');
if ($raw) {
    $json = json_decode($raw, true);
    if (is_array($json)) {
        $payload = $json;
    }
}

if (empty($payload) && !empty($_POST)) {
    $payload = $_POST;
}

$dealId = (int)($payload['deal_id'] ?? 0);
$purchaseRef = trim((string)($payload['purchase_ref'] ?? ''));
$customerId = isset($payload['customer_id']) ? (int)$payload['customer_id'] : null;
$customerName = isset($payload['customer_name']) ? trim((string)$payload['customer_name']) : null;
$amount = isset($payload['amount']) && $payload['amount'] !== '' ? (float)$payload['amount'] : null;
$currency = isset($payload['currency']) ? trim((string)$payload['currency']) : null;

$response = [
    'ok' => false,
    'notified' => false,
    'skipped' => false,
    'reason' => null,
];

if ($dealId <= 0 || $purchaseRef === '') {
    $response['skipped'] = true;
    $response['reason'] = 'invalid_payload';
    echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

// Log simples em arquivo
try {
    $logDir = __DIR__ . '/logs/nd_purchases';
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0775, true);
    }

    $ui = null;
    try {
        if (function_exists('getUserInfo') && isLogged()) {
            $ui = getUserInfo();
        }
    } catch (Throwable $e) {}

    $logData = [
        'time' => date('c'),
        'deal_id' => $dealId,
        'purchase_ref' => $purchaseRef,
        'customer_id' => (int)($customerId ?? ($ui['user_id'] ?? 0)),
        'customer_name' => (string)($customerName ?? ($ui['user_name'] ?? '')),
        'amount' => $amount,
        'currency' => $currency,
        'ip' => (string)($_SERVER['REMOTE_ADDR'] ?? ''),
        'ua' => (string)($_SERVER['HTTP_USER_AGENT'] ?? ''),
    ];

    $logFile = $logDir . '/' . date('Y-m-d') . '.jsonl';
    @file_put_contents($logFile, json_encode($logData, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . "\n", FILE_APPEND);
} catch (Throwable $e) {
    // silencioso
}

try {
    if (function_exists('nd_whatsapp_notify_merchant_purchase')) {
        $result = nd_whatsapp_notify_merchant_purchase(
            $connect,
            $dealId,
            $purchaseRef,
            $customerId,
            $customerName,
            $amount,
            $currency
        );

        if (!empty($result['ok'])) {
            $response['ok'] = true;
            $response['notified'] = true;
        } else {
            $response['ok'] = false;
            $response['notified'] = false;
            $response['skipped'] = (bool)($result['skipped'] ?? false);
            $response['reason'] = $result['reason'] ?? null;
        }
    } else {
        $response['skipped'] = true;
        $response['reason'] = 'whatsapp_helper_missing';
    }
} catch (Throwable $e) {
    $response['skipped'] = true;
    $response['reason'] = 'exception';
}

echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
