<?php
// controllers/report-offer.php
// JSON endpoint to receive "Reportar oferta" forms and forward via e-mail.
//
// Expected POST:
// - item (deal_id)    : string/int
// - reason            : string
// - details           : string (optional)
//
// Response: { ok: true } or { ok:false, error: "message" }
//
// Safe defaults: no database required. If PHPMailer is present (vendor/autoload.php or globally
// available), it will be used; otherwise falls back to mail().

header('Content-Type: application/json; charset=UTF-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');

// --- tiny helper ---
function json_out($ok, $msg = '', $extra = []){
  $out = array_merge(['ok' => (bool)$ok], $msg ? ['error' => $msg] : [], $extra);
  echo json_encode($out);
  exit;
}

// Only POST
if (strtoupper($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  http_response_code(405);
  json_out(false, 'Method Not Allowed');
}

// basic anti-abuse: 1 report/minute by IP+item (in memory file lock)
$ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
$item = trim((string)($_POST['item'] ?? ''));
$lockKey = sys_get_temp_dir() . '/nd_report_' . md5($ip.'|'.$item);
if (file_exists($lockKey) && filemtime($lockKey) > (time()-60)) {
  json_out(false, 'Aguarde alguns segundos e tente novamente.');
}
@touch($lockKey); @chmod($lockKey, 0600);

// Sanitization
$reason = trim((string)($_POST['reason'] ?? ''));
$details = trim((string)($_POST['details'] ?? ''));
if ($item === '') { json_out(false, 'Oferta não identificada.'); }
$allowed = ['inapropriado','expirado','link_quebrado','fraude','outro'];
if (!in_array($reason, $allowed, true)) { $reason = 'outro'; }
if (strlen($details) > 4000) { $details = substr($details, 0, 4000).'…'; }

// Try to capture logged user data (best-effort, adapt to your session schema)
@session_start();
$user_id    = $_SESSION['uid']          ?? $_SESSION['user']['id']    ?? null;
$user_name  = $_SESSION['name']         ?? $_SESSION['user']['name']  ?? $_SESSION['user']['fullname'] ?? '';
$user_email = $_SESSION['email']        ?? $_SESSION['user']['email'] ?? '';

// Context
$ua      = $_SERVER['HTTP_USER_AGENT'] ?? '';
$referer = $_SERVER['HTTP_REFERER']    ?? '';
$when    = date('Y-m-d H:i:s');

// Compose message
$toAddress = 'contato@ndmidia.com.br';  // <- alvo pedido pelo cliente
$fromAddr  = 'no-reply@conecta.ndmidia.com.br';
$subject   = "Denúncia de oferta #{$item} – {$reason}";
$plain     = "Uma nova denúncia foi enviada.\n\n".
             "Oferta (deal_id): {$item}\n".
             "Motivo: {$reason}\n".
             "Detalhes:\n{$details}\n\n".
             "Usuário: ".($user_name ?: '(anônimo)')."\n".
             "E-mail: ".($user_email ?: '(não informado)')."\n".
             "User ID: ".($user_id ?: '(n/d)')."\n\n".
             "IP: {$ip}\nUA: {$ua}\nOrigem: {$referer}\n".
             "Quando: {$when}\n";

$html = '<html><body style="font-family:system-ui,Segoe UI,Roboto,Arial,sans-serif;color:#222">'
      . '<h2 style="margin:0 0 12px">Nova denúncia de oferta</h2>'
      . '<table cellpadding="6" cellspacing="0" style="border-collapse:collapse">'
      . '<tr><td><b>Oferta (deal_id)</b></td><td>#'.htmlspecialchars($item).'</td></tr>'
      . '<tr><td><b>Motivo</b></td><td>'.htmlspecialchars($reason).'</td></tr>'
      . '<tr><td valign="top"><b>Detalhes</b></td><td><pre style="white-space:pre-wrap">'.htmlspecialchars($details).'</pre></td></tr>'
      . '<tr><td><b>Usuário</b></td><td>'.htmlspecialchars($user_name ?: '(anônimo)').'</td></tr>'
      . '<tr><td><b>E-mail</b></td><td>'.htmlspecialchars($user_email ?: '(não informado)').'</td></tr>'
      . '<tr><td><b>User ID</b></td><td>'.htmlspecialchars($user_id ?: '(n/d)').'</td></tr>'
      . '<tr><td><b>IP</b></td><td>'.htmlspecialchars($ip).'</td></tr>'
      . '<tr><td><b>User-Agent</b></td><td>'.htmlspecialchars($ua).'</td></tr>'
      . '<tr><td><b>Origem</b></td><td>'.htmlspecialchars($referer).'</td></tr>'
      . '<tr><td><b>Quando</b></td><td>'.htmlspecialchars($when).'</td></tr>'
      . '</table>'
      . '</body></html>';

// Send using PHPMailer if available; else mail()
$sent = false;
$err  = '';

if (!$sent) {
  // Try autoload
  try {
    if (file_exists(__DIR__.'/../vendor/autoload.php')) {
      require_once __DIR__.'/../vendor/autoload.php';
    }
    if (class_exists('\\PHPMailer\\PHPMailer\\PHPMailer')) {
      $mail = new \PHPMailer\PHPMailer\PHPMailer(true);
      // Transport: try default (sendmail/mail) to avoid needing SMTP creds
      $mail->isMail();
      $mail->setFrom($fromAddr, 'Cupons – Report');
      $mail->addAddress($toAddress);
      if ($user_email) $mail->addReplyTo($user_email, $user_name ?: $user_email);
      $mail->Subject = $subject;
      $mail->isHTML(true);
      $mail->Body    = $html;
      $mail->AltBody = $plain;
      $mail->CharSet = 'UTF-8';
      $mail->send();
      $sent = true;
    }
  } catch (\Throwable $e) {
    $err = $e->getMessage();
  }
}

if (!$sent) {
  // Fallback: native mail()
  $headers = [];
  $headers[] = 'MIME-Version: 1.0';
  $headers[] = 'Content-type: text/html; charset=UTF-8';
  $headers[] = 'From: Cupons – Report <'.$fromAddr.'>';
  if ($user_email) $headers[] = 'Reply-To: '.$user_name.' <'.$user_email.'>';
  $sent = @mail($toAddress, '=?UTF-8?B?'.base64_encode($subject).'?=', $html, implode("\r\n", $headers));
}

if (!$sent) {
  json_out(false, $err ?: 'Não foi possível enviar o e-mail agora.');
}

json_out(true);
