<?php
declare(strict_types=1);

/**
 * Dashboard do associado
 * - Utiliza layout do dashboard do anunciante
 * - Mostra métricas e QR Code do associado logado
 */

if (session_status() !== PHP_SESSION_ACTIVE) {
    @session_start();
}

if (!defined('ND_ASSOCIATE_DASHBOARD')) {
    define('ND_ASSOCIATE_DASHBOARD', true);
}

require_once __DIR__ . '/../core.php';
require_once __DIR__ . '/functions.php';
if (is_file(__DIR__ . '/../init_lang.php')) {
    require_once __DIR__ . '/../init_lang.php';
}

// Conexão PDO segura
$connect = $connect ?? null;
try {
    if (!($connect instanceof PDO) && function_exists('connect')) {
        if (isset($database)) {
            $connect = @connect($database);
        }
        if (!$connect) {
            $connect = @connect();
        }
    }
} catch (Throwable $e) {
    // segue com $connect nulo; view lida com valores padrão
}

// Gate de autenticação próprio para associados
$associateId = (int)($_SESSION['associate_id'] ?? 0);
if (empty($_SESSION['associate_signedin']) || $associateId <= 0) {
    if (!headers_sent()) {
        header('Location: associate_login.php');
    }
    exit;
}

// Dados básicos
$associate           = [];
$totalClicksToday    = 0;
$totalClicksLast30   = 0;
$totalClicks         = 0;
$totalUniqueClicks   = 0;
$topAssociates      = [];
$rankingMonthLabel  = date('m/Y');
$chartLabels         = [];
$chartValues         = [];
$qrText              = '';
$qrImg               = '';
$avgClicksLast30     = 0.0;
$bestDayLabel        = '';
$bestDayValue        = 0;
$lastClickAt         = '';
$lastClickAtBr       = '';

try {
    if ($connect instanceof PDO) {
        // Carrega dados do associado
        $stmt = $connect->prepare('SELECT * FROM associates WHERE associate_id = :id LIMIT 1');
        $stmt->execute([':id' => $associateId]);
        $associate = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];

        // Total de cliques (lifetime)
        $stmtTotal = $connect->prepare('SELECT COUNT(*) AS total FROM associate_clicks WHERE associate_id = :id');
        $stmtTotal->execute([':id' => $associateId]);
        $rt = $stmtTotal->fetch(PDO::FETCH_ASSOC);
        if ($rt && isset($rt['total'])) {
            $totalClicks = (int)$rt['total'];
        }

        // Cliques nos últimos 30 dias
        $stmt30 = $connect->prepare(
            'SELECT COUNT(*) AS total
               FROM associate_clicks
              WHERE associate_id = :id
                AND click_time >= DATE_SUB(NOW(), INTERVAL 30 DAY)'
        );
        $stmt30->execute([':id' => $associateId]);
        $r30 = $stmt30->fetch(PDO::FETCH_ASSOC);
        if ($r30 && isset($r30['total'])) {
            $totalClicksLast30 = (int)$r30['total'];
        }

        // Cliques nos últimos 7 dias
        $stmt7 = $connect->prepare(
            'SELECT COUNT(*) AS total
               FROM associate_clicks
              WHERE associate_id = :id
                AND click_time >= DATE_SUB(NOW(), INTERVAL 7 DAY)'
        );
        $stmt7->execute([':id' => $associateId]);
        $r7 = $stmt7->fetch(PDO::FETCH_ASSOC);
        if ($r7 && isset($r7['total'])) {
            $totalClicksLast7 = (int)$r7['total'];
        }


        // Cliques de hoje
        $stmtToday = $connect->prepare(
            'SELECT COUNT(*) AS total
               FROM associate_clicks
              WHERE associate_id = :id
                AND DATE(click_time) = CURDATE()'
        );
        $stmtToday->execute([':id' => $associateId]);
        $rToday = $stmtToday->fetch(PDO::FETCH_ASSOC);
        if ($rToday && isset($rToday['total'])) {
            $totalClicksToday = (int)$rToday['total'];
        }

        // Cliques únicos (por IP)
        $stmtU = $connect->prepare(
            'SELECT COUNT(DISTINCT click_ip) AS total
               FROM associate_clicks
              WHERE associate_id = :id'
        );
        $stmtU->execute([':id' => $associateId]);
        $rU = $stmtU->fetch(PDO::FETCH_ASSOC);
        if ($rU && isset($rU['total'])) {
            $totalUniqueClicks = (int)$rU['total'];
        }

        // Gráfico: cliques por dia (últimos 30 dias)
        $sql = 'SELECT DATE(click_time) AS day, COUNT(*) AS total
                  FROM associate_clicks
                 WHERE associate_id = :id
                   AND click_time >= DATE_SUB(NOW(), INTERVAL 365 DAY)
              GROUP BY DATE(click_time)
              ORDER BY day ASC';
        $stmtDay = $connect->prepare($sql);
        $stmtDay->execute([':id' => $associateId]);
        $rows = $stmtDay->fetchAll(PDO::FETCH_ASSOC);

        foreach ($rows as $row) {
            $chartLabels[] = (string)($row['day'] ?? '');
            $chartValues[] = (int)($row['total'] ?? 0);
        }

        // Estatísticas avançadas (média e melhor dia)
        if (!empty($chartValues)) {
            $sum = array_sum($chartValues);
            $countDays = count($chartValues);
            if ($countDays > 0) {
                $avgClicksLast30 = round($sum / $countDays, 2);
            }

            $maxVal = max($chartValues);
            $idxMax = array_search($maxVal, $chartValues, true);
            if ($idxMax !== false && isset($chartLabels[$idxMax])) {
                $bestDayLabel = $chartLabels[$idxMax];
                $bestDayValue = $maxVal;
            }
        }

        // Último clique registrado
        $stmtLast = $connect->prepare(
            'SELECT MAX(click_time) AS last_time
               FROM associate_clicks
              WHERE associate_id = :id'
        );
        $stmtLast->execute([':id' => $associateId]);
        $rLast = $stmtLast->fetch(PDO::FETCH_ASSOC);
        if ($rLast && !empty($rLast['last_time'])) {
            $lastClickAt = (string)$rLast['last_time'];
            $ts = strtotime($lastClickAt);
            if ($ts) {
                $lastClickAtBr = date('d/m/Y H:i', $ts);
            } else {
                $lastClickAtBr = $lastClickAt;
            }
        }

        // Ranking: Top 10 associados por cliques no mês atual
        try {
            if ($connect instanceof PDO) {
                $dbRank = $connect;
                $sqlRank = "SELECT a.associate_id,
                                   a.associate_name,
                                   a.associate_code,
                                   COUNT(c.click_id) AS total_clicks_month
                            FROM associates a
                            LEFT JOIN associate_clicks c
                              ON c.associate_id = a.associate_id
                             AND c.click_time >= DATE_FORMAT(CURDATE(), '%Y-%m-01')
                            GROUP BY a.associate_id
                            ORDER BY total_clicks_month DESC, a.associate_name ASC
                            LIMIT 10";
                $stmtRank = $dbRank->prepare($sqlRank);
                $stmtRank->execute();
                $topAssociates = $stmtRank->fetchAll(PDO::FETCH_ASSOC) ?: [];
            }
        } catch (Throwable $e) {
            $topAssociates = [];
        }

        // QR Code para este associado
        if (!empty($associate['associate_code'] ?? '')) {
            $qrTextBase = rtrim(SITE_URL, '/') . '/qr.php?code=' . urlencode((string)$associate['associate_code']);
            $qrText     = $qrTextBase;
            $qrImg      = 'https://api.qrserver.com/v1/create-qr-code/?size=220x220&data=' . urlencode($qrTextBase);
        }
    }
} catch (Throwable $e) {
    // Em caso de erro, mantém valores padrão
}

/* Render */
require __DIR__ . '/header.php';
require __DIR__ . '/views/associate-home.view.php';
require __DIR__ . '/footer.php';
