<?php
declare(strict_types=1);

/**
 * Dashboard: Meus Dados do Associado
 * - Permite ao associado ver e atualizar seus próprios dados
 * - Inclui upload de foto (avatar) simples em /storage/associates
 */

if (session_status() !== PHP_SESSION_ACTIVE) {
    @session_start();
}

if (!defined('ND_ASSOCIATE_DASHBOARD')) {
    define('ND_ASSOCIATE_DASHBOARD', true);
}

require_once __DIR__ . '/../core.php';
require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/../admin/functions.php';
if (is_file(__DIR__ . '/../init_lang.php')) {
    require_once __DIR__ . '/../init_lang.php';
}

// DEBUG: loga sessão e ID do associado para troubleshooting
try {
    $debugDir = __DIR__ . '/../storage/logs';
    if (!is_dir($debugDir)) {
        @mkdir($debugDir, 0775, true);
    }
    $debugFile = $debugDir . '/associate_profile_debug.log';
    $dbg = date('Y-m-d H:i:s') . " associate_profile.php\n";
    $dbg .= "SESSION: " . print_r($_SESSION, true) . "\n";
    @file_put_contents($debugFile, $dbg . "\n----------------------\n", FILE_APPEND);
} catch (Throwable $e) {
    // silenciosamente ignora falha de log
}


// Garante que é um associado logado
$associateId = isset($_SESSION['associate_id']) ? (int)$_SESSION['associate_id'] : 0;
if (empty($_SESSION['associate_signedin']) || $associateId <= 0) {
    if (!headers_sent()) {
        header('Location: associate_login.php');
    }
    exit;
}

$errors  = [];
$success = '';

// Conexão PDO
$connect = $connect ?? null;
try {
    if (!$connect && function_exists('connect')) {
        $connect = connect();
    }
} catch (\Throwable $e) {
    $connect = null;
}

$associate = [];

// 1) Tenta carregar via helper padrão (usado no admin)
if (function_exists('get_associate_per_id')) {
    try {
        $row = get_associate_per_id($associateId);
        if (is_array($row)) {
            $associate = $row;
        }
    } catch (\Throwable $e) {
        // silencia erro aqui; vamos tentar fallback logo abaixo
        $associate = [];
    }
}

// 2) Fallback: se ainda não veio nada, usa a conexão PDO local
if ((!$associate || !isset($associate['associate_id'])) && $connect instanceof PDO) {
    try {
        $stmt = $connect->prepare('SELECT * FROM associates WHERE associate_id = :id LIMIT 1');
        $stmt->execute([':id' => $associateId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (is_array($row)) {
            $associate = $row;
        }
    } catch (\Throwable $e) {
        // Mantém array vazio em caso de erro
        $associate = [];
    }
}

// DEBUG extra: grava no log como ficou o array do associado
try {
    $debugDir = __DIR__ . '/../storage/logs';
    if (!is_dir($debugDir)) {
        @mkdir($debugDir, 0775, true);
    }
    $debugFile = $debugDir . '/associate_profile_debug.log';
    $dbg = date('Y-m-d H:i:s') . " AFTER_LOAD associate_profile.php\n";
    $dbg .= "associateId: " . $associateId . "\n";
    $dbg .= "associate array: " . print_r($associate, true) . "\n";
    @file_put_contents($debugFile, $dbg . "\n----------------------\n", FILE_APPEND);
} catch (\Throwable $e) {
    // ignora falha de log
}


// Processa envio do formulário
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $connect instanceof PDO) {
    $name   = trim((string)($_POST['associate_name']   ?? ''));
    $email  = trim((string)($_POST['associate_email']  ?? ''));
    $phone  = trim((string)($_POST['associate_phone']  ?? ''));
    $city   = trim((string)($_POST['associate_city']   ?? ''));
    $state  = trim((string)($_POST['associate_state']  ?? ''));
    $group  = trim((string)($_POST['associate_group']  ?? ''));

    if ($name === '') {
        $errors[] = 'O nome é obrigatório.';
    }

    if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Informe um e-mail válido.';
    }

    if (!$errors) {
        try {
            $stmt = $connect->prepare('
                UPDATE associates
                   SET associate_name  = :name,
                       associate_email = :email,
                       associate_phone = :phone,
                       associate_city  = :city,
                       associate_state = :state,
                       associate_group = :grp
                 WHERE associate_id   = :id
                 LIMIT 1
            ');
            $stmt->execute([
                ':name'  => $name,
                ':email' => $email,
                ':phone' => $phone,
                ':city'  => $city,
                ':state' => $state,
                ':grp'   => $group,
                ':id'    => $associateId,
            ]);

            // Atualiza nome em sessão
            $_SESSION['associate_name'] = $name;

            $success = 'Dados atualizados com sucesso!';
        } catch (\Throwable $e) {
            $errors[] = 'Erro ao atualizar seus dados. Tente novamente.';
        }
    }

    // Upload de avatar (opcional)
    if (!empty($_FILES['associate_avatar']['name'])
        && is_uploaded_file($_FILES['associate_avatar']['tmp_name'])) {

        $tmp  = $_FILES['associate_avatar']['tmp_name'];
        $nameFile = (string) ($_FILES['associate_avatar']['name'] ?? '');
        $ext  = strtolower(pathinfo($nameFile, PATHINFO_EXTENSION));

        if (!in_array($ext, ['jpg', 'jpeg', 'png'], true)) {
            $errors[] = 'A foto precisa ser JPG ou PNG.';
        } else {
            // Diretório físico para armazenar avatares dos associados em storage/associates
            $baseDir   = realpath(dirname(__DIR__)); // raiz do projeto conecta.ndmidia.com.br
            $uploadDir = $baseDir . '/storage/associates';
            if (!is_dir($uploadDir)) {
                @mkdir($uploadDir, 0775, true);
            }
            if (is_dir($uploadDir) && is_writable($uploadDir)) {
                // Remove arquivos anteriores com outras extensões
                foreach (['jpg','jpeg','png'] as $e) {
                    $old = $uploadDir . '/avatar_' . $associateId . '.' . $e;
                    if (is_file($old)) {
                        @unlink($old);
                    }
                }
                $dest = $uploadDir . '/avatar_' . $associateId . '.' . $ext;
                if (!@move_uploaded_file($tmp, $dest)) {
                    $errors[] = 'Não foi possível salvar a foto. Verifique as permissões da pasta raiz do projeto.';
                } else {
                    $success = $success ?: 'Dados atualizados com sucesso!';
                }
            } else {
                $errors[] = 'Pasta raiz do projeto não é gravável.';
            }
        }
    }

    // Recarrega dados após update
    if (function_exists('get_associate_per_id')) {
        $row = get_associate_per_id($associateId);
        if (is_array($row)) {
            $associate = $row;
        }
    } elseif ($connect instanceof PDO) {
        try {
            $stmt = $connect->prepare('SELECT * FROM associates WHERE associate_id = :id LIMIT 1');
            $stmt->execute([':id' => $associateId]);
            $associate = (array) $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (\Throwable $e) {
            // mantém versão anterior
        }
    }

}

// Render
require __DIR__ . '/header.php';
require __DIR__ . '/views/associate-profile.view.php';
require __DIR__ . '/footer.php';