<?php
declare(strict_types=1);
/*--------------------*/
// Dashboard - Meus Cupons (por oferta)
/*--------------------*/

if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }

require_once __DIR__ . '/../core.php';
require_once __DIR__ . '/functions.php';
if (is_file(__DIR__ . '/../init_lang.php')) { require_once __DIR__ . '/../init_lang.php'; }

/* ========= Conexão PDO segura ========= */
$connect = $connect ?? null;
try {
    if (!$connect && function_exists('connect')) {
        global $database;
        if (isset($database)) {
            $connect = @connect($database);
        }
    }
} catch (Throwable $e) {
    $connect = null;
}

/* ========= Gates (somente anunciante logado) ========= */
if (!nd_is_logged($connect)) {
    if (!headers_sent()) { header('Location: ./login.php'); }
    exit;
}
if (!nd_is_seller()) {
    if (!headers_sent()) { header('Location: ./denied.php'); }
    exit;
}

/* ========= Dados do usuário ========= */
$userProfile = [];
$userDetails = [];
try { if (function_exists('getUserInfo')) { $userProfile = (array)getUserInfo(); } } catch (Throwable $e) {}
try {
    if (function_exists('getUserInfoById') && !empty($userProfile['user_id'])) {
        $userDetails = (array)getUserInfoById((int)$userProfile['user_id']);
    }
} catch (Throwable $e) {}

$userId = (int)($userProfile['user_id'] ?? 0);
if ($userId <= 0) {
    if (!headers_sent()) { header('Location: ./login.php'); }
    exit;
}

/* ========= Seller do usuário ========= */
$sellerTitle = null;
$sellerSlug  = null;
if ($connect) {
    try {
        $st = $connect->prepare("
            SELECT seller_title, seller_slug
            FROM sellers
            WHERE seller_user = :uid AND seller_status = 1
            LIMIT 1
        ");
        $st->bindValue(':uid', $userId, PDO::PARAM_INT);
        $st->execute();
        $row = $st->fetch(PDO::FETCH_ASSOC);
        if ($row) {
            $sellerTitle = trim((string)($row['seller_title'] ?? ''));
            $sellerSlug  = trim((string)($row['seller_slug'] ?? ''));
        }
    } catch (Throwable $e) {
        $sellerTitle = null;
        $sellerSlug  = null;
    }
}

/* ========= Loja ligada a esse seller ========= */
/* Usa somente stores: slug ou título iguais ao seller */
$storeId = null;
$storeSlug = null;
if ($connect && ($sellerTitle || $sellerSlug)) {
    try {
        $st = $connect->prepare("
            SELECT store_id, store_slug
            FROM stores
            WHERE (
                    (:slug  <> '' AND store_slug  = :slug)
                 OR (:title <> '' AND store_title = :title)
                  )
            LIMIT 1
        ");
        $st->bindValue(':slug',  (string)($sellerSlug  ?? ''),  PDO::PARAM_STR);
        $st->bindValue(':title', (string)($sellerTitle ?? ''), PDO::PARAM_STR);
        $st->execute();
        $storeRow = $st->fetch(PDO::FETCH_ASSOC);
        if ($storeRow) {
            $storeId   = (int)$storeRow['store_id'];
            $storeSlug = (string)($storeRow['store_slug'] ?? '');
        }
    } catch (Throwable $e) {
        $storeId   = null;
        $storeSlug = null;
    }
}

/* ========= Ofertas do anunciante (para o select) ========= */
/*
 * Regra:
 *  - deal_status = 1
 *  - E ( deal_author = user_id
 *      OU deal_store = store_id da loja do seller
 *      OU deal_store = store_slug da loja do seller )
 *
 *  Para Panificadora Lucas:
 *   - sellers.seller_slug  = 'panificadora-lucas'
 *   - stores.store_id      = 12
 *   - deals.deal_store     = 12  (Café da Manhã na Padaria)
 */
$deals = [];
if ($connect) {
    try {

        $baseSql = "
            SELECT d.deal_id, d.deal_title
            FROM deals d
            WHERE d.deal_status = 1
              AND (
                    d.deal_author = :uid
        ";

        if ($storeId !== null || $storeSlug) {
            $baseSql .= "
                    OR d.deal_store = :store_id
                    OR d.deal_store = :store_slug
            ";
        }

        $baseSql .= "
              )
            ORDER BY d.deal_created DESC
        ";

        $st = $connect->prepare($baseSql);
        $st->bindValue(':uid', $userId, PDO::PARAM_INT);

        if ($storeId !== null || $storeSlug) {
            // se não tiver loja vinculada, só filtra por author
            $sid  = ($storeId   !== null) ? $storeId   : 0;
            $ssl  = ($storeSlug !== null) ? $storeSlug : '';
            $st->bindValue(':store_id',   $sid, PDO::PARAM_INT);
            $st->bindValue(':store_slug', $ssl, PDO::PARAM_STR);
        }

        $st->execute();
        $deals = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
    } catch (Throwable $e) {
        $deals = [];
    }
}

/* ========= Processa criação de cupom ========= */
$errors  = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST'
    && isset($_POST['action'])
    && $_POST['action'] === 'create_coupon') {

    $deal_id          = isset($_POST['deal_id']) ? (int)$_POST['deal_id'] : 0;
    $coupon_title     = trim($_POST['coupon_title'] ?? '');
    $coupon_code      = trim($_POST['coupon_code'] ?? '');
    $discount_percent = isset($_POST['discount_percent']) ? (int)$_POST['discount_percent'] : 0;
    $quantity_raw     = trim((string)($_POST['quantity'] ?? ''));
    $quantity         = ($quantity_raw === '') ? null : (int)$quantity_raw;

    if ($deal_id <= 0)        { $errors[] = 'Selecione uma oferta.'; }
    if ($coupon_title === '') { $errors[] = 'Informe um título para o cupom.'; }
    if ($coupon_code === '')  { $errors[] = 'Informe um código de cupom.'; }
    if ($discount_percent <= 0 || $discount_percent > 100) {
        $errors[] = 'O desconto deve estar entre 1% e 100%.';
    }

    // Valida se a oferta pertence a este anunciante (mesma regra do select)
    if (!$errors && $connect) {
        try {
            $sql = "
                SELECT d.deal_id
                FROM deals d
                WHERE d.deal_id = :id
                  AND d.deal_status = 1
                  AND (
                        d.deal_author = :uid
            ";
            if ($storeId !== null || $storeSlug) {
                $sql .= "
                        OR d.deal_store = :store_id
                        OR d.deal_store = :store_slug
                ";
            }
            $sql .= "
                  )
                LIMIT 1
            ";
            $st = $connect->prepare($sql);
            $st->bindValue(':id',  $deal_id, PDO::PARAM_INT);
            $st->bindValue(':uid', $userId,  PDO::PARAM_INT);

            if ($storeId !== null || $storeSlug) {
                $sid = ($storeId !== null) ? $storeId : 0;
                $ssl = ($storeSlug !== null) ? $storeSlug : '';
                $st->bindValue(':store_id',   $sid, PDO::PARAM_INT);
                $st->bindValue(':store_slug', $ssl, PDO::PARAM_STR);
            }

            $st->execute();
            if (!$st->fetch(PDO::FETCH_ASSOC)) {
                $errors[] = 'A oferta selecionada não pertence a este anunciante.';
            }
        } catch (Throwable $e) {
            $errors[] = 'Erro ao validar a oferta.';
        }
    }

    // Verifica duplicidade de código
    if (!$errors && $connect) {
        try {
            $sql = "SELECT coupon_id
                    FROM deal_coupons
                    WHERE user_id = :uid AND coupon_code = :code
                    LIMIT 1";
            $st = $connect->prepare($sql);
            $st->bindValue(':uid',  $userId,      PDO::PARAM_INT);
            $st->bindValue(':code', $coupon_code, PDO::PARAM_STR);
            $st->execute();
            if ($st->fetch(PDO::FETCH_ASSOC)) {
                $errors[] = 'Você já possui um cupom com este código.';
            }
        } catch (Throwable $e) {
            // ignora erro aqui
        }
    }

    // Insere cupom
    if (!$errors && $connect) {
        try {
            $sql = "INSERT INTO deal_coupons
                        (deal_id, user_id, coupon_title, coupon_code,
                         discount_percent, quantity, status, created_at)
                    VALUES
                        (:deal_id, :user_id, :coupon_title, :coupon_code,
                         :discount_percent, :quantity, 1, NOW())";
            $st = $connect->prepare($sql);
            $st->bindValue(':deal_id',          $deal_id,          PDO::PARAM_INT);
            $st->bindValue(':user_id',          $userId,           PDO::PARAM_INT);
            $st->bindValue(':coupon_title',     $coupon_title,     PDO::PARAM_STR);
            $st->bindValue(':coupon_code',      $coupon_code,      PDO::PARAM_STR);
            $st->bindValue(':discount_percent', $discount_percent, PDO::PARAM_INT);
            if ($quantity === null) {
                $st->bindValue(':quantity', null, PDO::PARAM_NULL);
            } else {
                $st->bindValue(':quantity', $quantity, PDO::PARAM_INT);
            }
                        $st->execute();
            $success = true;

            // WhatsApp automático para comerciante (opcional)
            if (function_exists('nd_notify_store_coupon_created')) {
                try {
                    nd_notify_store_coupon_created(
                        $connect,
                        (int) $deal_id,
                        (string) $coupon_title,
                        (string) $coupon_code,
                        (int) $discount_percent,
                        $quantity
                    );
                } catch (Throwable $e) {
                    if (function_exists('nd_whatsapp_log')) {
                        nd_whatsapp_log('Erro ao notificar cupom criado', [
                            'deal_id' => $deal_id,
                            'coupon_code' => $coupon_code,
                            'error' => $e->getMessage()
                        ]);
                    }
                }
            }
        } catch (Throwable $e) {
            $errors[] = 'Erro ao salvar o cupom.';
        }
    }
}

/* ========= Lista de cupons do anunciante ========= */
$coupons = [];
if ($connect) {
    try {
        $sql = "SELECT dc.*, d.deal_title,
                       (
                         SELECT COUNT(*) FROM deal_coupon_usages u
                         WHERE u.coupon_id = dc.coupon_id
                       ) AS times_used
                FROM deal_coupons dc
                LEFT JOIN deals d ON dc.deal_id = d.deal_id
                WHERE dc.user_id = :uid
                ORDER BY dc.coupon_id DESC";
        $st = $connect->prepare($sql);
        $st->bindValue(':uid', $userId, PDO::PARAM_INT);
        $st->execute();
        $coupons = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
    } catch (Throwable $e) {
        $coupons = [];
    }
}



/* ========= Resumo de conversão por oferta ========= */
$dealSummary = [];
if ($connect) {
    try {
        $sql = "
            SELECT
                dc.deal_id,
                d.deal_title,
                COUNT(dc.coupon_id) AS coupons_count,
                SUM(CASE WHEN dc.quantity IS NULL OR dc.quantity = 0 THEN 0 ELSE dc.quantity END) AS total_quantity,
                SUM(CASE WHEN dc.quantity IS NULL OR dc.quantity = 0 THEN 1 ELSE 0 END) AS unlimited_count,
                COUNT(u.usage_id) AS total_used
            FROM deal_coupons dc
            LEFT JOIN deals d ON dc.deal_id = d.deal_id
            LEFT JOIN deal_coupon_usages u ON u.coupon_id = dc.coupon_id
            WHERE dc.user_id = :uid
            GROUP BY dc.deal_id, d.deal_title
            ORDER BY d.deal_title ASC
        ";
        $st = $connect->prepare($sql);
        $st->bindValue(':uid', $userId, PDO::PARAM_INT);
        $st->execute();
        $dealSummary = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
    } catch (Throwable $e) {
        $dealSummary = [];
    }
}

/* ========= Traduções básicas ========= */
$translation = (isset($translation) && is_array($translation)) ? $translation : [];
$translation += [
    'nd_coupons_title'        => 'Meus Cupons',
    'nd_coupons_new'          => 'Criar novo cupom',
    'nd_coupons_deal'         => 'Oferta',
    'nd_coupons_coupon_title' => 'Título do cupom',
    'nd_coupons_coupon_code'  => 'Código do cupom',
    'nd_coupons_discount'     => 'Desconto (%)',
    'nd_coupons_quantity'     => 'Quantidade (opcional)',
    'nd_coupons_save'         => 'Salvar cupom',
    'nd_coupons_list'         => 'Cupons cadastrados',
];

/* ========= Render ========= */
require __DIR__ . '/header.php';
require __DIR__ . '/views/coupons.view.php';
require __DIR__ . '/footer.php';
