<?php
declare(strict_types=1);

/**
 * Header bootstrap do Dashboard
 * - Garante sessão
 * - Força idioma PT-BR no painel (mantém seletor funcionando se você quiser)
 * - Carrega arquivo de linguagem em /languages
 * - Expõe $translation, $langCode, $langDir, $siteLocale
 */

require_once __DIR__ . '/../core.php'; // mantém seu core

// 1) Sessão
if (session_status() !== PHP_SESSION_ACTIVE) {
  @session_start();
}

// >>> Garanta $forced ANTES de qualquer uso (evita Undefined variable)
$forced = $forced ?? 'pt_br';

// 2) Caminhos
$ROOT     = dirname(__DIR__);     // raiz do projeto
$LANG_DIR = $ROOT . '/languages'; // a sua pasta de idiomas
$PARTIALS = $ROOT . '/partials';  // a sua pasta de parciais

// 3) Idioma — FORCE PT-BR no dashboard
//    Se quiser permitir troca, comente a linha que define $forced e use a query ?lang=xx

/* ND_LANG_SELECTOR_OVERRIDE_START */
if (!headers_sent()) {
    if (isset($_GET['lang']) && $_GET['lang'] !== '') {
        $sel = preg_replace('/[^a-zA-Z_\-]/', '', (string)$_GET['lang']);
        setcookie('nd_lang', $sel, time() + 3600*24*365, '/');
        $_COOKIE['nd_lang'] = $sel;
    }
}
if (isset($_COOKIE['nd_lang']) && $_COOKIE['nd_lang'] !== '') {
    $forced = ''; // cookie sobrepõe o forçado padrão
}
/* ND_LANG_SELECTOR_OVERRIDE_END */

// opções comuns: 'pt', 'pt_br', 'en', 'es'
$allowed   = ['pt', 'pt_br', 'en', 'es', 'es_es'];
$requested = isset($_GET['lang']) ? strtolower(trim((string)$_GET['lang'])) : '';

// se não veio via GET mas há cookie e o "forçado" foi liberado, usa cookie
if ($requested === '' && $forced === '' && isset($_COOKIE['nd_lang']) && $_COOKIE['nd_lang'] !== '') {
  $requested = strtolower(trim((string)$_COOKIE['nd_lang']));
  $requested = preg_replace('/[^a-zA-Z_\-]/', '', $requested);
} else {
  $requested = preg_replace('/[^a-zA-Z_\-]/', '', $requested);
}

if ($forced) {
  $langCode = $forced;
  $_SESSION['lang'] = $forced;
} else {
  $langCode = $_SESSION['lang'] ?? '';
  if ($requested && in_array($requested, $allowed, true)) {
    $langCode = $requested;
    $_SESSION['lang'] = $requested;
  }
  if ($langCode === '' || !in_array($langCode, $allowed, true)) {
    $langCode = 'pt_br';
    $_SESSION['lang'] = $langCode;
  }
}

// 4) Loader de traduções
function nd_load_lang(string $code, string $langDir): array {
  // normaliza aliases
  $map = [
    'pt'    => 'pt_br',
    'pt_br' => 'pt_br',
    'en'    => 'en',
    'es'    => 'es',
    'es_es' => 'es',
  ];
  $code = $map[$code] ?? $code;

  $phpFile  = $langDir . '/' . $code . '.php';
  $jsonFile = $langDir . '/' . $code . '.json';

  // PHP (retornando array) tem prioridade
  if (is_file($phpFile)) {
    $t = include $phpFile;
    return is_array($t) ? $t : [];
  }

  // JSON como fallback
  if (is_file($jsonFile)) {
    $raw = file_get_contents($jsonFile);
    $j = json_decode((string)$raw, true);
    return is_array($j) ? $j : [];
  }

  // Último fallback: vazio
  return [];
}

$translation  = nd_load_lang($langCode, $LANG_DIR);

// compat: se o projeto usar também $translations (plural)
$translations = isset($translations) && is_array($translations) ? $translations : [];

// Mescla “novo” em “antigo” sem perder chaves existentes
$translation = $translation + $translations;

// 5) Direção do idioma (todos os seus são LTR)
$langDir    = 'ltr';
$siteLocale = ($langCode === 'pt' || $langCode === 'pt_br') ? 'pt-BR' :
              (($langCode === 'es' || $langCode === 'es_es') ? 'es-ES' : 'en-US');

// 6) Exponha timezone BR por padrão (opcional)
if (function_exists('date_default_timezone_set')) {
  @date_default_timezone_set('America/Sao_Paulo');
}

// (Opcional) disponibiliza caminho do parcial do seletor se você usa
$__lang_partial_path = $PARTIALS . '/lang-switch.php';
if (is_file($__lang_partial_path)) {
  // não inclui aqui; o menu.view move/renderiza quando necessário
}

// Por fim, render da “view” do header
require __DIR__ . '/views/header.view.php';
