<?php
declare(strict_types=1);

/**
 * Cadastro de Anunciantes (Dashboard)
 * - Cria usuário na tabela `users`
 * - Cria registro vinculado na tabela `sellers`
 * - Faz login automático e redireciona para home.php
 */

require_once __DIR__ . '/../core.php';

if (session_status() !== PHP_SESSION_ACTIVE) {
  @session_start();
}

/* Garante conexão PDO em $connect */
try {
  if (!isset($connect) || !$connect) {
    if (function_exists('connect')) {
      if (isset($database)) {
        $connect = @connect($database);
      } else {
        $connect = @connect();
      }
    }
  }
} catch (Throwable $e) {
  $connect = $connect ?? null;
}

/* Fallbacks de tradução usados aqui */
$translation = (isset($translation) && is_array($translation)) ? $translation : [];
$translation += [
  'tr_seller_signup_title'   => 'Criar conta de anunciante',
  'tr_seller_signup_sub'     => 'Cadastre-se para anunciar suas ofertas no ND Conecta.',
  'tr_fullname'              => 'Nome completo',
  'tr_email'                 => 'E-mail',
  'tr_password'              => 'Senha',
  'tr_password_confirm'      => 'Confirmar senha',
  'tr_store_name'            => 'Nome da loja',
  'tr_store_desc'            => 'Descrição da loja',
  'tr_store_website'         => 'Site da loja (opcional)',
  'tr_btn_create_account'    => 'Criar conta',
  'tr_have_account'          => 'Já tem uma conta?',
  'tr_go_login'              => 'Ir para o login',
  'tr_error_required'        => 'Preencha todos os campos obrigatórios.',
  'tr_error_email_invalid'   => 'E-mail inválido.',
  'tr_error_email_exists'    => 'Já existe uma conta com este e-mail.',
  'tr_error_pass_short'      => 'A senha deve ter pelo menos 6 caracteres.',
  'tr_error_pass_match'      => 'As senhas não coincidem.',
  'tr_error_db'              => 'Erro ao salvar seus dados. Tente novamente.',
];

if (!function_exists('tr_reg')) {
  function tr_reg(string $k, string $fallback = ''): string {
    $t = $GLOBALS['translation'] ?? [];
    $val = (string)($t[$k] ?? $fallback);
    return htmlspecialchars($val, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
  }
}

/* SEO básico */
$titleSeoHeader       = getSeoTitle($translation['tr_seller_signup_title'], 'Cadastro');
$descriptionSeoHeader = getSeoDescription($translation['tr_seller_signup_sub']);

/* Se já está logado, manda direto pro painel */
try {
  if (function_exists('isLogged') && isLogged()) {
    if (!headers_sent()) {
      header('Location: home.php');
    }
    exit;
  }
} catch (Throwable $e) {
  // segue fluxo normal
}

/* Valores padrão para repopular formulário */
$name           = trim((string)($_POST['name']           ?? ''));
$email          = trim((string)($_POST['email']          ?? ''));
$store_title    = trim((string)($_POST['store_title']    ?? ''));
$store_desc     = trim((string)($_POST['store_desc']     ?? ''));
$store_website  = trim((string)($_POST['store_website']  ?? ''));
$errors         = [];
$success        = '';

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'POST') {
  $pass   = (string)($_POST['password']  ?? '');
  $pass2  = (string)($_POST['password2'] ?? '');

  // Validações básicas
  if ($name === '' || $email === '' || $pass === '' || $pass2 === '' || $store_title === '') {
    $errors[] = $translation['tr_error_required'];
  }

  if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $errors[] = $translation['tr_error_email_invalid'];
  }

  if (strlen($pass) < 6) {
    $errors[] = $translation['tr_error_pass_short'];
  }

  if ($pass !== $pass2) {
    $errors[] = $translation['tr_error_pass_match'];
  }

  // Se ainda não há erros, valida e persiste no banco
  if (empty($errors) && $connect instanceof PDO) {
    try {
      // E-mail já existe?
      $st = $connect->prepare("SELECT 1 FROM users WHERE user_email = :email LIMIT 1");
      $st->execute([':email' => strtolower($email)]);
      if ($st->fetchColumn()) {
        $errors[] = $translation['tr_error_email_exists'];
      }
    } catch (Throwable $e) {
      $errors[] = $translation['tr_error_db'];
    }
  }

  if (empty($errors) && $connect instanceof PDO) {
    try {
      $connect->beginTransaction();

      // Cria usuário
      $hash = hash('sha512', $pass);
      $insUser = $connect->prepare("INSERT INTO users (user_email, user_password, user_name, user_status, user_verified) VALUES (:email, :pwd, :name, 1, 0)");
      $insUser->execute([
        ':email' => strtolower($email),
        ':pwd'   => $hash,
        ':name'  => $name,
      ]);
      $userId = (int)$connect->lastInsertId();

      // Cria seller vinculado
      require_once __DIR__ . '/../classes/slugify.php';
      $slug = convertSlug($store_title);

      $insSeller = $connect->prepare("INSERT INTO sellers (seller_title, seller_description, seller_website, seller_logo, seller_slug, seller_status, seller_user) VALUES (:title, :descr, :website, 'avatar.png', :slug, 1, :uid)");
      $insSeller->execute([
        ':title'   => $store_title,
        ':descr'   => $store_desc,
        ':website' => $store_website !== '' ? $store_website : null,
        ':slug'    => $slug,
        ':uid'     => $userId,
      ]);

      $sellerId = (int)$connect->lastInsertId();

      $connect->commit();

      // Faz login automático
      $_SESSION['signedin']   = true;
      $_SESSION['user_email'] = strtolower($email);
      $_SESSION['user_name']  = $name;
      if ($sellerId > 0) {
        $_SESSION['seller_id'] = $sellerId;
      }

      if (!headers_sent()) {
        header('Location: home.php');
      }
      exit;

    } catch (Throwable $e) {
      if ($connect instanceof PDO && $connect->inTransaction()) {
        $connect->rollBack();
      }
      $errors[] = $translation['tr_error_db'];
    }
  }
}

/* Render padrão do dashboard */
require __DIR__ . '/header.php';
require __DIR__ . '/views/register.view.php';
require __DIR__ . '/footer.php';
