<?php
declare(strict_types=1);

/*--------------------*/
// Dashboard Reports (seller)
/*--------------------*/

if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }

require_once __DIR__ . '/../core.php';
require_once __DIR__ . '/functions.php';
if (is_file(__DIR__ . '/../init_lang.php')) { require_once __DIR__ . '/../init_lang.php'; }

/* Conexão PDO segura */
$connect = $connect ?? null;
try {
  if (!isset($connect) || !$connect) {
    if (function_exists('connect')) {
      if (isset($database)) { $connect = @connect($database); }
      if (!$connect)        { $connect = @connect(); }
    }
  }
} catch (Throwable $e) {}

/* Guards */
if (!function_exists('nd_is_logged')) {
  function nd_is_logged($conn = null): bool {
    if (function_exists('isLogged')) {
      try { return isset($conn) ? (bool)isLogged($conn) : (bool)isLogged(); }
      catch (Throwable $e) { return false; }
    }
    return !empty($_SESSION['user_id']) || !empty($_SESSION['user']);
  }
}
if (!function_exists('nd_is_seller')) {
  function nd_is_seller(): bool {
    if (function_exists('isSeller')) {
      try { return (bool)isSeller(); } catch (Throwable $e) { return false; }
    }
    return false;
  }
}

/* Gate */
if (!nd_is_logged($connect)) {
  if (!headers_sent()) { header('Location: ./login.php'); }
  exit;
}
if (!nd_is_seller()) {
  if (!headers_sent()) { header('Location: ./denied.php'); }
  exit;
}

/* User */
$userProfile = [];
try { if (function_exists('getUserInfo')) $userProfile = (array)getUserInfo(); } catch (Throwable $e) {}
$userId = (int)($userProfile['user_id'] ?? ($_SESSION['user_id'] ?? 0));

/* --------------------
 * Public report token (per store)
 * - Allows seller to generate a shareable link without login
 * - Uses SITE_URL constant from config.php
 * -------------------- */
if (!function_exists('nd_reports_ensure_token_table')) {
  function nd_reports_ensure_token_table($connect): void {
    if (!$connect) return;
    try {
      $q = $connect->query("SHOW TABLES LIKE 'nd_report_tokens'");
      $exists = (bool)($q && $q->fetch(PDO::FETCH_NUM));
      if ($exists) return;
      $connect->exec(
        "CREATE TABLE nd_report_tokens (\n".
        "  id INT UNSIGNED NOT NULL AUTO_INCREMENT,\n".
        "  token CHAR(64) NOT NULL,\n".
        "  store_id INT NOT NULL,\n".
        "  user_id INT NOT NULL,\n".
        "  is_revoked TINYINT(1) NOT NULL DEFAULT 0,\n".
        "  created_at DATETIME NOT NULL,\n".
        "  last_used_at DATETIME NULL,\n".
        "  revoked_at DATETIME NULL,\n".
        "  PRIMARY KEY (id),\n".
        "  UNIQUE KEY uniq_token (token),\n".
        "  KEY idx_store_user (store_id, user_id),\n".
        "  KEY idx_revoked (is_revoked)\n".
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci"
      );
    } catch (Throwable $e) {
      // silently ignore
    }
  }
}

if (!function_exists('nd_reports_get_or_create_token')) {
  function nd_reports_get_or_create_token($connect, int $userId, int $storeId, bool $regenerate = false): ?string {
    if (!$connect || $userId <= 0 || $storeId <= 0) return null;
    nd_reports_ensure_token_table($connect);

    try {
      if (!$regenerate) {
        $st = $connect->prepare("SELECT token FROM nd_report_tokens WHERE user_id = :uid AND store_id = :sid AND is_revoked = 0 ORDER BY id DESC LIMIT 1");
        $st->execute([':uid' => $userId, ':sid' => $storeId]);
        $tok = (string)($st->fetchColumn() ?: '');
        if ($tok !== '') return $tok;
      }

      // revoke existing tokens for store
      $rv = $connect->prepare("UPDATE nd_report_tokens SET is_revoked = 1, revoked_at = NOW() WHERE user_id = :uid AND store_id = :sid AND is_revoked = 0");
      $rv->execute([':uid' => $userId, ':sid' => $storeId]);

      // create new token
      $token = bin2hex(random_bytes(24)); // 48 bytes hex => 48? actually 24->48 chars
      $ins = $connect->prepare("INSERT INTO nd_report_tokens (token, store_id, user_id, is_revoked, created_at) VALUES (:t, :sid, :uid, 0, NOW())");
      $ins->execute([':t' => $token, ':sid' => $storeId, ':uid' => $userId]);
      return $token;
    } catch (Throwable $e) {
      return null;
    }
  }
}

$publicReportUrl = '';
$publicReportError = '';

/* Date range */
$tz = new DateTimeZone('America/Sao_Paulo');
$today = new DateTime('now', $tz);

$startStr = isset($_GET['from']) ? (string)$_GET['from'] : '';
$endStr   = isset($_GET['to'])   ? (string)$_GET['to']   : '';
$start = null; $end = null;

try {
  if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $startStr)) $start = new DateTime($startStr . ' 00:00:00', $tz);
  if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $endStr))   $end   = new DateTime($endStr   . ' 23:59:59', $tz);
} catch (Throwable $e) {}

if (!$start) { $start = (clone $today)->modify('-30 days')->setTime(0,0,0); }
if (!$end)   { $end   = (clone $today)->setTime(23,59,59); }

$filterStoreId = isset($_GET['store_id']) ? (int)$_GET['store_id'] : 0;

/* Stores do usuário */
$stores = [];
try {
  if ($connect && $userId > 0) {
    $st = $connect->prepare("SELECT store_id, store_title FROM stores WHERE store_user_id = :uid AND store_status = 1 ORDER BY store_title ASC");
    $st->execute([':uid' => $userId]);
    $stores = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  }
} catch (Throwable $e) { $stores = []; }

$storeIds = array_map(fn($r) => (int)($r['store_id'] ?? 0), $stores);
$storeIds = array_values(array_filter($storeIds, fn($v) => $v > 0));

if ($filterStoreId > 0 && !in_array($filterStoreId, $storeIds, true)) {
  $filterStoreId = 0; // não pertence ao user
}

$hasEventsTable = false;
try {
  if ($connect) {
    $q = $connect->query("SHOW TABLES LIKE 'nd_events'");
    $hasEventsTable = (bool)($q && $q->fetch(PDO::FETCH_NUM));
  }
} catch (Throwable $e) { $hasEventsTable = false; }

/* Handle public link generation (POST) */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['nd_public_link'])) {
  $storeForToken = (int)($_POST['store_id'] ?? 0);
  $regenerate = isset($_POST['regenerate']) && (string)$_POST['regenerate'] === '1';

  if ($storeForToken <= 0) {
    $publicReportError = 'Selecione uma loja específica para gerar o link público.';
  } elseif (!in_array($storeForToken, $storeIds, true)) {
    $publicReportError = 'Loja inválida para sua conta.';
  } else {
    $tok = nd_reports_get_or_create_token($connect, $userId, $storeForToken, $regenerate);
    if ($tok) {
      $base = defined('SITE_URL') ? rtrim((string)SITE_URL, '/') : '';
      $publicReportUrl = ($base !== '' ? $base : '') . '/public-report.php?t=' . urlencode($tok);
    } else {
      $publicReportError = 'Não foi possível gerar o link público agora. Tente novamente.';
    }
  }
}

/* Dados */
$summary = [];
$topDeals = [];
$overall = ['sessions' => 0, 'associates' => 0];
$topAssociates = [];

// Month-to-date mini summary (sellable headline)
$mtd = ['views' => 0, 'clicks' => 0, 'copies' => 0];
if ($connect && $hasEventsTable && !empty($storeIds)) {
  try {
    $idsM = ($filterStoreId > 0) ? [$filterStoreId] : $storeIds;
    $inM  = implode(',', array_fill(0, count($idsM), '?'));
    $monthStart = new DateTime('first day of this month 00:00:00', $tz);
    $monthEnd   = (clone $today);

    $sql = "SELECT\n".
           "  SUM(event_name = 'view_deal') AS views,\n".
           "  SUM(event_name = 'click_deal_link') AS clicks,\n".
           "  SUM(event_name = 'copy_coupon') AS copies\n".
           "FROM nd_events\n".
           "WHERE store_id IN ($inM)\n".
           "  AND created_at BETWEEN ? AND ?";
    $st = $connect->prepare($sql);
    $i = 1;
    foreach ($idsM as $sid) { $st->bindValue($i++, (int)$sid, PDO::PARAM_INT); }
    $st->bindValue($i++, $monthStart->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->bindValue($i++, $monthEnd->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->execute();
    $row = $st->fetch(PDO::FETCH_ASSOC) ?: [];
    $mtd = [
      'views'  => (int)($row['views'] ?? 0),
      'clicks' => (int)($row['clicks'] ?? 0),
      'copies' => (int)($row['copies'] ?? 0),
    ];
  } catch (Throwable $e) {
    $mtd = ['views' => 0, 'clicks' => 0, 'copies' => 0];
  }
}

if ($connect && $hasEventsTable && !empty($storeIds)) {
  $ids = ($filterStoreId > 0) ? [$filterStoreId] : $storeIds;
  $in  = implode(',', array_fill(0, count($ids), '?'));

  try {
    $sql = "SELECT event_name,
                   COUNT(*) AS total,
                   COUNT(DISTINCT session_id) AS sessions,
                   COUNT(DISTINCT associate_id) AS associates
            FROM nd_events
            WHERE store_id IN ($in)
              AND created_at BETWEEN ? AND ?
            GROUP BY event_name
            ORDER BY total DESC";
    $st = $connect->prepare($sql);
    $i = 1;
    foreach ($ids as $sid) { $st->bindValue($i++, (int)$sid, PDO::PARAM_INT); }
    $st->bindValue($i++, $start->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->bindValue($i++, $end->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->execute();
    $summary = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  } catch (Throwable $e) { $summary = []; }

  try {
    $sql = "SELECT d.deal_id, d.deal_title,
                   SUM(e.event_name = 'view_deal') AS views,
                   SUM(e.event_name = 'click_deal_link') AS clicks,
                   SUM(e.event_name = 'copy_coupon') AS coupon_copies
            FROM nd_events e
            LEFT JOIN deals d ON d.deal_id = e.deal_id
            WHERE e.store_id IN ($in)
              AND e.created_at BETWEEN ? AND ?
            GROUP BY d.deal_id, d.deal_title
            ORDER BY clicks DESC, views DESC
            LIMIT 20";
    $st = $connect->prepare($sql);
    $i = 1;
    foreach ($ids as $sid) { $st->bindValue($i++, (int)$sid, PDO::PARAM_INT); }
    $st->bindValue($i++, $start->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->bindValue($i++, $end->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->execute();
    $topDeals = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  } catch (Throwable $e) { $topDeals = []; }

  // Resumo geral (sessions e associados únicos no período)
  try {
    $sql = "SELECT COUNT(DISTINCT session_id) AS sessions,
                   COUNT(DISTINCT associate_id) AS associates
            FROM nd_events
            WHERE store_id IN ($in)
              AND created_at BETWEEN ? AND ?";
    $st = $connect->prepare($sql);
    $i = 1;
    foreach ($ids as $sid) { $st->bindValue($i++, (int)$sid, PDO::PARAM_INT); }
    $st->bindValue($i++, $start->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->bindValue($i++, $end->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->execute();
    $overall = $st->fetch(PDO::FETCH_ASSOC) ?: ['sessions' => 0, 'associates' => 0];
  } catch (Throwable $e) {
    $overall = ['sessions' => 0, 'associates' => 0];
  }

  // Top associados (quem mais gerou tráfego/ações para as suas ofertas)
  try {
    $sql = "SELECT e.associate_id, a.associate_name, a.associate_city, a.associate_state, a.associate_group,
                   SUM(e.event_name = 'view_deal') AS views,
                   SUM(e.event_name = 'click_deal_link') AS clicks,
                   SUM(e.event_name = 'copy_coupon') AS coupon_copies,
                   COUNT(*) AS total_events
            FROM nd_events e
            LEFT JOIN associates a ON a.associate_id = e.associate_id
            WHERE e.store_id IN ($in)
              AND e.associate_id IS NOT NULL
              AND e.created_at BETWEEN ? AND ?
            GROUP BY e.associate_id, a.associate_name, a.associate_city, a.associate_state, a.associate_group
            ORDER BY clicks DESC, views DESC, total_events DESC
            LIMIT 20";
    $st = $connect->prepare($sql);
    $i = 1;
    foreach ($ids as $sid) { $st->bindValue($i++, (int)$sid, PDO::PARAM_INT); }
    $st->bindValue($i++, $start->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->bindValue($i++, $end->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->execute();
    $topAssociates = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  } catch (Throwable $e) {
    $topAssociates = [];
  }
}

/* Render */
require __DIR__ . '/header.php';
require __DIR__ . '/views/reports.view.php';
require __DIR__ . '/footer.php';
