<?php
declare(strict_types=1);

/*--------------------*/
// Dashboard Reports Export CSV (seller)
/*--------------------*/

if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }

require_once __DIR__ . '/../core.php';
require_once __DIR__ . '/functions.php';
if (is_file(__DIR__ . '/../init_lang.php')) { require_once __DIR__ . '/../init_lang.php'; }

/* Conexão PDO segura */
$connect = $connect ?? null;
try {
  if (!isset($connect) || !$connect) {
    if (function_exists('connect')) {
      if (isset($database)) { $connect = @connect($database); }
      if (!$connect)        { $connect = @connect(); }
    }
  }
} catch (Throwable $e) {}

/* Guards */
if (!function_exists('nd_is_logged')) {
  function nd_is_logged($conn = null): bool {
    if (function_exists('isLogged')) {
      try { return isset($conn) ? (bool)isLogged($conn) : (bool)isLogged(); }
      catch (Throwable $e) { return false; }
    }
    return !empty($_SESSION['user_id']) || !empty($_SESSION['user']);
  }
}
if (!function_exists('nd_is_seller')) {
  function nd_is_seller(): bool {
    if (function_exists('isSeller')) {
      try { return (bool)isSeller(); } catch (Throwable $e) { return false; }
    }
    return false;
  }
}

if (!nd_is_logged($connect) || !nd_is_seller()) {
  http_response_code(403);
  exit('Acesso negado');
}

/* User */
$userProfile = [];
try { if (function_exists('getUserInfo')) $userProfile = (array)getUserInfo(); } catch (Throwable $e) {}
$userId = (int)($userProfile['user_id'] ?? ($_SESSION['user_id'] ?? 0));

/* Date range */
$tz = new DateTimeZone('America/Sao_Paulo');
$today = new DateTime('now', $tz);

$startStr = isset($_GET['from']) ? (string)$_GET['from'] : '';
$endStr   = isset($_GET['to'])   ? (string)$_GET['to']   : '';
$start = null; $end = null;

try {
  if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $startStr)) $start = new DateTime($startStr . ' 00:00:00', $tz);
  if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $endStr))   $end   = new DateTime($endStr   . ' 23:59:59', $tz);
} catch (Throwable $e) {}

if (!$start) { $start = (clone $today)->modify('-30 days')->setTime(0,0,0); }
if (!$end)   { $end   = (clone $today)->setTime(23,59,59); }

$filterStoreId = isset($_GET['store_id']) ? (int)$_GET['store_id'] : 0;

/* Stores do usuário */
$stores = [];
try {
  if ($connect && $userId > 0) {
    $st = $connect->prepare("SELECT store_id, store_title FROM stores WHERE store_user_id = :uid AND store_status = 1 ORDER BY store_title ASC");
    $st->execute([':uid' => $userId]);
    $stores = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  }
} catch (Throwable $e) { $stores = []; }

$storeIds = array_map(fn($r) => (int)($r['store_id'] ?? 0), $stores);
$storeIds = array_values(array_filter($storeIds, fn($v) => $v > 0));

if ($filterStoreId > 0 && !in_array($filterStoreId, $storeIds, true)) {
  $filterStoreId = 0; // não pertence ao user
}

$hasEventsTable = false;
try {
  if ($connect) {
    $q = $connect->query("SHOW TABLES LIKE 'nd_events'");
    $hasEventsTable = (bool)($q && $q->fetch(PDO::FETCH_NUM));
  }
} catch (Throwable $e) { $hasEventsTable = false; }

$summary = [];
$topDeals = [];
$overall = ['sessions' => 0, 'associates' => 0];
$topAssociates = [];

if ($connect && $hasEventsTable && !empty($storeIds)) {
  $ids = ($filterStoreId > 0) ? [$filterStoreId] : $storeIds;
  $in  = implode(',', array_fill(0, count($ids), '?'));

  try {
    $sql = "SELECT event_name,
                   COUNT(*) AS total,
                   COUNT(DISTINCT session_id) AS sessions,
                   COUNT(DISTINCT associate_id) AS associates
            FROM nd_events
            WHERE store_id IN ($in)
              AND created_at BETWEEN ? AND ?
            GROUP BY event_name";
    $st = $connect->prepare($sql);
    $i = 1;
    foreach ($ids as $sid) { $st->bindValue($i++, (int)$sid, PDO::PARAM_INT); }
    $st->bindValue($i++, $start->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->bindValue($i++, $end->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->execute();
    $summary = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  } catch (Throwable $e) { $summary = []; }

  try {
    $sql = "SELECT d.deal_id, d.deal_title,
                   SUM(e.event_name = 'view_deal') AS views,
                   SUM(e.event_name = 'click_deal_link') AS clicks,
                   SUM(e.event_name = 'copy_coupon') AS coupon_copies
            FROM nd_events e
            LEFT JOIN deals d ON d.deal_id = e.deal_id
            WHERE e.store_id IN ($in)
              AND e.created_at BETWEEN ? AND ?
            GROUP BY d.deal_id, d.deal_title
            ORDER BY clicks DESC, views DESC
            LIMIT 200";
    $st = $connect->prepare($sql);
    $i = 1;
    foreach ($ids as $sid) { $st->bindValue($i++, (int)$sid, PDO::PARAM_INT); }
    $st->bindValue($i++, $start->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->bindValue($i++, $end->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->execute();
    $topDeals = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  } catch (Throwable $e) { $topDeals = []; }

  try {
    $sql = "SELECT COUNT(DISTINCT session_id) AS sessions,
                   COUNT(DISTINCT associate_id) AS associates
            FROM nd_events
            WHERE store_id IN ($in)
              AND created_at BETWEEN ? AND ?";
    $st = $connect->prepare($sql);
    $i = 1;
    foreach ($ids as $sid) { $st->bindValue($i++, (int)$sid, PDO::PARAM_INT); }
    $st->bindValue($i++, $start->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->bindValue($i++, $end->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->execute();
    $overall = $st->fetch(PDO::FETCH_ASSOC) ?: ['sessions' => 0, 'associates' => 0];
  } catch (Throwable $e) { $overall = ['sessions' => 0, 'associates' => 0]; }

  try {
    $sql = "SELECT e.associate_id, a.associate_name, a.associate_city, a.associate_state, a.associate_group,
                   SUM(e.event_name = 'view_deal') AS views,
                   SUM(e.event_name = 'click_deal_link') AS clicks,
                   SUM(e.event_name = 'copy_coupon') AS coupon_copies
            FROM nd_events e
            LEFT JOIN associates a ON a.associate_id = e.associate_id
            WHERE e.store_id IN ($in)
              AND e.associate_id IS NOT NULL
              AND e.created_at BETWEEN ? AND ?
            GROUP BY e.associate_id, a.associate_name, a.associate_city, a.associate_state, a.associate_group
            ORDER BY clicks DESC, views DESC
            LIMIT 200";
    $st = $connect->prepare($sql);
    $i = 1;
    foreach ($ids as $sid) { $st->bindValue($i++, (int)$sid, PDO::PARAM_INT); }
    $st->bindValue($i++, $start->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->bindValue($i++, $end->format('Y-m-d H:i:s'), PDO::PARAM_STR);
    $st->execute();
    $topAssociates = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  } catch (Throwable $e) { $topAssociates = []; }
}

$map = [];
foreach ($summary as $row) {
  $map[(string)($row['event_name'] ?? '')] = $row;
}
$views  = (int)($map['view_deal']['total'] ?? 0);
$clicks = (int)($map['click_deal_link']['total'] ?? 0);
$copies = (int)($map['copy_coupon']['total'] ?? 0);
$ctr = ($views > 0) ? round(($clicks / $views) * 100, 1) : 0.0;
$copyRate = ($clicks > 0) ? round(($copies / $clicks) * 100, 1) : 0.0;

/* CSV */
$filename = 'relatorio_' . date('Ymd_His') . '.csv';
header('Content-Type: text/csv; charset=utf-8');
header('Content-Disposition: attachment; filename="' . $filename . '"');

$out = fopen('php://output', 'w');

// BOM para Excel (pt-BR)
fprintf($out, "\xEF\xBB\xBF");

$sep = ';';

function fputcsv_sc($handle, array $row, string $sep = ';') {
  // compat com Excel Brasil
  fputcsv($handle, $row, $sep);
}

$storeTitle = 'Todas as lojas';
if ($filterStoreId > 0) {
  foreach ($stores as $s) {
    if ((int)$s['store_id'] === $filterStoreId) { $storeTitle = (string)$s['store_title']; break; }
  }
}

fputcsv_sc($out, ['Seção','Item','Valor 1','Valor 2','Valor 3'], $sep);

fputcsv_sc($out, ['Filtro','Loja',$storeTitle,'',''], $sep);
fputcsv_sc($out, ['Filtro','Período',$start->format('d/m/Y') . ' até ' . $end->format('d/m/Y'),'',''], $sep);

fputcsv_sc($out, ['Resumo','Sessões únicas',(string)((int)($overall['sessions'] ?? 0)),'',''], $sep);
fputcsv_sc($out, ['Resumo','Associados únicos',(string)((int)($overall['associates'] ?? 0)),'',''], $sep);
fputcsv_sc($out, ['Resumo','Visualizações (views)',(string)$views,'',''], $sep);
fputcsv_sc($out, ['Resumo','Cliques no CTA',(string)$clicks,'CTR',str_replace('.',',',(string)$ctr) . '%'], $sep);
fputcsv_sc($out, ['Resumo','Cupom copiado',(string)$copies,'Taxa (cliques→cupom)',str_replace('.',',',(string)$copyRate) . '%'], $sep);

fputcsv_sc($out, ['','','','',''], $sep);

fputcsv_sc($out, ['Top ofertas','Oferta','Views','Cliques','Cupom'], $sep);
foreach ($topDeals as $d) {
  fputcsv_sc($out, ['Top ofertas', (string)($d['deal_title'] ?? ('#'.$d['deal_id'])), (string)((int)($d['views'] ?? 0)), (string)((int)($d['clicks'] ?? 0)), (string)((int)($d['coupon_copies'] ?? 0))], $sep);
}

fputcsv_sc($out, ['','','','',''], $sep);

fputcsv_sc($out, ['Top associados','Associado','Cidade/UF','Grupo','Cliques'], $sep);
foreach ($topAssociates as $a) {
  $city = trim((string)($a['associate_city'] ?? ''));
  $st = trim((string)($a['associate_state'] ?? ''));
  $loc = $city;
  if ($st !== '') $loc = ($loc !== '' ? ($loc.' / '.$st) : $st);
  $grp = (string)($a['associate_group'] ?? '');
  fputcsv_sc($out, ['Top associados', (string)($a['associate_name'] ?? ('#'.$a['associate_id'])), $loc, $grp, (string)((int)($a['clicks'] ?? 0))], $sep);
}

fclose($out);
exit;
