<?php
declare(strict_types=1);

/*--------------------*/
// Descrição: ND Cupons 
// Autor: ND Tecnologia 
// Autor URI: https://www.ndmidia.com.br 
/*--------------------*/

// Núcleo e funções (caminhos corretos a partir de /dashboard/)
require_once __DIR__ . '/../core.php';
require_once __DIR__ . '/functions.php';

// i18n (se existir)
if (is_file(__DIR__ . '/../init_lang.php')) {
  require_once __DIR__ . '/../init_lang.php';
}

// Garante sessão
if (session_status() !== PHP_SESSION_ACTIVE) {
  @session_start();
}

/* Conexão PDO segura */
$connect = $connect ?? null;
try {
  if (!isset($connect) || !$connect) {
    if (function_exists('connect')) {
      if (isset($database)) {
        $connect = @connect($database);
      }
      if (!$connect) {
        $connect = @connect();
      }
    }
  }
} catch (Throwable $e) {
  // se falhar, $connect poderá ser null; os guards abaixo cobrem
}

/* Helpers “seguros” para evitar fatals se funções não existirem */
$nd_is_logged = function($conn = null): bool {
  if (function_exists('isLogged')) {
    try { return isset($conn) ? (bool)isLogged($conn) : (bool)isLogged(); }
    catch (Throwable $e) { return false; }
  }
  return !empty($_SESSION['user_id']) || !empty($_SESSION['user']);
};

$nd_is_seller = function(): bool {
  if (function_exists('isSeller')) {
    try { return (bool)isSeller(); }
    catch (Throwable $e) { return false; }
  }
  // Se não existir a função, trate como não-seller
  return false;
};

$nd_is_expired = function(): bool {
  if (function_exists('isExpiredSubscription')) {
    try { return (bool)isExpiredSubscription(); }
    catch (Throwable $e) { return false; }
  }
  // Se não houver controle de expiração, considere “não expirado”
  return false;
};

$nd_get_user_info = function(): array {
  try {
    if (function_exists('getUserInfo')) return (array)getUserInfo();
  } catch (Throwable $e) {}
  return [];
};

$nd_get_user_info_by_id = function(int $id): array {
  try {
    if (function_exists('getUserInfoById')) return (array)getUserInfoById($id);
  } catch (Throwable $e) {}
  return [];
};

$nd_get_plan_by_id = function($pdo, $planId): array {
  try {
    if (function_exists('getPlanById') && $pdo && $planId) {
      $p = getPlanById($pdo, (int)$planId);
      return is_array($p) ? $p : [];
    }
  } catch (Throwable $e) {}
  return [];
};

$nd_get_seller_by_id = function(int $id): array {
  try {
    if (function_exists('getSellerById')) return (array)getSellerById($id);
  } catch (Throwable $e) {}
  return [];
};

/* ====== Fluxo principal ====== */
if (!$nd_is_logged($connect)) {
  // não logado → login do dashboard
  if (!headers_sent()) {
    header('Location: ./login.php');
  }
  exit;
}

// perfil atual
$userProfile = $nd_get_user_info();
$userDetails = [];
if (!empty($userProfile['user_id'])) {
  $userDetails = $nd_get_user_info_by_id((int)$userProfile['user_id']);
}

// exige papel de seller
if (!$nd_is_seller()) {
  if (!headers_sent()) {
    header('Location: ./denied.php');
  }
  exit;
}

// bloqueia se assinatura expirada
if ($nd_is_expired()) {
  if (!headers_sent()) {
    header('Location: ./home.php');
  }
  exit;
}

// configurações do plano (se houver)
$userPlanSettings = [];
if (!empty($userDetails['user_plan'])) {
  $userPlanSettings = $nd_get_plan_by_id($connect, (int)$userDetails['user_plan']);
}

// precisa permitir “custom store” no plano?
if (!empty($userPlanSettings) && array_key_exists('plan_customstore', $userPlanSettings)) {
  if ((int)$userPlanSettings['plan_customstore'] !== 1) {
    if (!headers_sent()) {
      header('Location: ./upgrade.php');
    }
    exit;
  }
}


// ---- Gerenciamento de lojas do anunciante (multi-loja) ----
$storeErrors  = [];
$storeSuccess = null;
$myStores     = [];

try {
  if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['store_action'])) {
    $action = $_POST['store_action'] ?? '';
    $uid    = (int)($userProfile['user_id'] ?? 0);
    if ($uid > 0 && $connect) {
      $title    = trim($_POST['store_title'] ?? '');
      $desc     = trim($_POST['store_description'] ?? '');
      $featured = isset($_POST['store_featured']) ? (int)$_POST['store_featured'] : 0;
      $status   = isset($_POST['store_status']) ? (int)$_POST['store_status'] : 1;

      if ($action === 'create') {
        if ($title === '') {
          $storeErrors[] = 'O nome da loja é obrigatório.';
        } else {
          // gerar slug único
          $slug = function_exists('slugify_basic') ? slugify_basic($title) : strtolower(preg_replace('/[^a-z0-9]+/i','-',$title));
          $base = $slug; $i = 1;
          $st = $connect->prepare('SELECT store_id FROM stores WHERE store_slug = :slug LIMIT 1');
          while (true) {
            $st->execute([':slug'=>$slug]);
            if (!$st->fetch()) break;
            $slug = $base . '-' . $i;
            $i++;
          }

          // upload de logo (opcional)
          $image_new_name = '';
          if (!empty($_FILES['store_image']['name'])) {
            $file = $_FILES['store_image'];
            $ext  = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            $allowed = ['jpg','jpeg','png','gif'];
            if (in_array($ext, $allowed) && $file['size'] > 0 && $file['error'] == 0) {
              $image_new_name = md5(time().rand()) . '.' . $ext;
              $target_dir = __DIR__ . '/../images/';
              if (!is_dir($target_dir)) { @mkdir($target_dir, 0775, true); }
              @move_uploaded_file($file['tmp_name'], $target_dir . $image_new_name);
            }
          }

          // tenta inserir com coluna store_user_id
          try {
            $ins = $connect->prepare('INSERT INTO stores (store_title, store_seotitle, store_description, store_seodescription, store_featured, store_slug, store_image, store_url, store_status, store_user_id) VALUES (:t, :st, :d, :sd, :f, :s, :img, :url, :stt, :uid)');
            $ins->execute([
              ':t'   => $title,
              ':st'  => $title,
              ':d'   => $desc,
              ':sd'  => $desc,
              ':f'   => $featured,
              ':s'   => $slug,
              ':img' => $image_new_name,
              ':url' => '',
              ':stt' => $status,
              ':uid' => $uid
            ]);
          } catch (Throwable $e) {
            // fallback sem coluna store_user_id
            $ins = $connect->prepare('INSERT INTO stores (store_title, store_seotitle, store_description, store_seodescription, store_featured, store_slug, store_image, store_url, store_status) VALUES (:t, :st, :d, :sd, :f, :s, :img, :url, :stt)');
            $ins->execute([
              ':t'   => $title,
              ':st'  => $title,
              ':d'   => $desc,
              ':sd'  => $desc,
              ':f'   => $featured,
              ':s'   => $slug,
              ':img' => $image_new_name,
              ':url' => '',
              ':stt' => $status
            ]);
          }

          $storeSuccess = 'Loja criada com sucesso!';
        }
      } elseif ($action === 'update') {
        $store_id = (int)($_POST['store_id'] ?? 0);
        if ($store_id > 0) {
          // garantir propriedade (quando existir coluna store_user_id)
          $ok = false;
          try {
            $ck = $connect->prepare('SELECT store_id FROM stores WHERE store_id = :id AND store_user_id = :uid LIMIT 1');
            $ck->execute([':id'=>$store_id, ':uid'=>$uid]);
            $ok = (bool)$ck->fetch();
          } catch (Throwable $e) {
            $ok = true;
          }
          if ($ok) {
            // upload opcional de novo logo
            $image_new_name = '';
            if (!empty($_FILES['store_image']['name'])) {
              $file = $_FILES['store_image'];
              $ext  = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
              $allowed = ['jpg','jpeg','png','gif'];
              if (in_array($ext, $allowed) && $file['size'] > 0 && $file['error'] == 0) {
                $image_new_name = md5(time().rand()) . '.' . $ext;
                $target_dir = __DIR__ . '/../images/';
                if (!is_dir($target_dir)) { @mkdir($target_dir, 0775, true); }
                @move_uploaded_file($file['tmp_name'], $target_dir . $image_new_name);
              }
            }

            $sql = 'UPDATE stores SET store_title = :t, store_description = :d, store_featured = :f, store_status = :s';
            $params = [':t'=>$title, ':d'=>$desc, ':f'=>$featured, ':s'=>$status, ':id'=>$store_id];
            if ($image_new_name !== '') {
              $sql .= ', store_image = :img';
              $params[':img'] = $image_new_name;
            }
            $sql .= ' WHERE store_id = :id';
            $upd = $connect->prepare($sql);
            $upd->execute($params);
            $storeSuccess = 'Loja atualizada!';
          }
        }
      } elseif ($action === 'delete') {
        $store_id = (int)($_POST['store_id'] ?? 0);
        if ($store_id > 0) {
          $ok = false;
          try {
            $ck = $connect->prepare('SELECT store_id FROM stores WHERE store_id = :id AND store_user_id = :uid LIMIT 1');
            $ck->execute([':id'=>$store_id, ':uid'=>$uid]);
            $ok = (bool)$ck->fetch();
          } catch (Throwable $e) {
            $ok = true;
          }
          if ($ok) {
            $del = $connect->prepare('DELETE FROM stores WHERE store_id = :id');
            $del->execute([':id'=>$store_id]);
            $storeSuccess = 'Loja removida.';
          }
        }
      }
    }
  }
} catch (Throwable $e) {
  $storeErrors[] = 'Erro: ' . $e->getMessage();
}

if (!empty($userProfile['user_id']) && $connect) {
  $myStores = getStoresForUser($connect, (int)$userProfile['user_id']);
}
// ---- fim gerenciamento de lojas do anunciante ----

// detalhes do seller
$sellerDetails = [];
if (!empty($userProfile['user_id'])) {
  $sellerDetails = $nd_get_seller_by_id((int)$userProfile['user_id']);
}

/* Render (views do dashboard) */
require __DIR__ . '/header.php';
require __DIR__ . '/views/seller.view.php';
require __DIR__ . '/footer.php';