<?php
declare(strict_types=1);

/**
 * ND - Marca cupom como UTILIZADO (painel do comerciante)
 * Endpoint seguro para uso interno do dashboard.
 *
 * Recebe POST:
 *  - deal_id
 *  - coupon_code
 *  - coupon_title (opcional)
 *
 * Comportamento:
 *  - Exige anunciante logado
 *  - Registra log leve em /storage/coupons_used/YYYY-MM-DD.jsonl
 *  - Dispara notificação WhatsApp ao comerciante (opcional)
 */

if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }

require_once __DIR__ . '/../core.php';
require_once __DIR__ . '/functions.php';

header('Content-Type: application/json; charset=UTF-8');

$response = [
    'ok' => false,
    'notified' => false,
    'skipped' => false,
    'reason' => null,
];

$connect = $connect ?? null;
try {
    if (!$connect && function_exists('connect')) {
        global $database;
        if (isset($database)) {
            $connect = @connect($database);
        }
        if (!$connect) {
            $connect = @connect();
        }
    }
} catch (Throwable $e) {
    $connect = null;
}

if (!nd_is_logged($connect)) {
    $response['skipped'] = true;
    $response['reason'] = 'not_logged';
    echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

$dealId = isset($_POST['deal_id']) ? (int) $_POST['deal_id'] : 0;
$couponCode = isset($_POST['coupon_code']) ? trim((string) $_POST['coupon_code']) : '';
$couponTitle = isset($_POST['coupon_title']) ? trim((string) $_POST['coupon_title']) : null;

if ($dealId <= 0 || $couponCode === '') {
    $response['skipped'] = true;
    $response['reason'] = 'invalid_payload';
    echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

// Identificação do usuário do painel (se disponível)
$usedBy = '';
try {
    if (!empty($_SESSION['user_email'])) {
        $usedBy = (string) $_SESSION['user_email'];
    } elseif (!empty($_SESSION['user_id'])) {
        $usedBy = 'user#' . (string) $_SESSION['user_id'];
    }
} catch (Throwable $e) {
    $usedBy = '';
}



// Registro em tabela de usos de cupom (métrica de conversão)
if ($connect) {
    try {
        // Cria tabela se ainda não existir (compatibilidade retroativa)
        $ddl = "CREATE TABLE IF NOT EXISTS deal_coupon_usages (
            usage_id INT UNSIGNED NOT NULL AUTO_INCREMENT,
            coupon_id INT UNSIGNED NOT NULL,
            deal_id INT NOT NULL,
            user_id INT NOT NULL,
            used_by VARCHAR(150) DEFAULT NULL,
            used_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (usage_id),
            KEY idx_coupon_id (coupon_id),
            KEY idx_deal_id (deal_id),
            KEY idx_user_id (user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        $connect->exec($ddl);

        $userId = (int) ($_SESSION['user_id'] ?? 0);

        // Localiza o cupom pelo deal_id + código + usuário
        $couponId = null;
        $st = $connect->prepare("
            SELECT coupon_id
            FROM deal_coupons
            WHERE deal_id = :deal_id
              AND coupon_code = :coupon_code
              AND user_id = :user_id
            ORDER BY coupon_id DESC
            LIMIT 1
        ");
        $st->bindValue(':deal_id', $dealId, PDO::PARAM_INT);
        $st->bindValue(':coupon_code', $couponCode, PDO::PARAM_STR);
        $st->bindValue(':user_id', $userId, PDO::PARAM_INT);
        $st->execute();
        $row = $st->fetch(PDO::FETCH_ASSOC);
        if ($row && !empty($row['coupon_id'])) {
            $couponId = (int) $row['coupon_id'];
        }

        if ($couponId) {
            $st = $connect->prepare("
                INSERT INTO deal_coupon_usages (coupon_id, deal_id, user_id, used_by)
                VALUES (:coupon_id, :deal_id, :user_id, :used_by)
            ");
            $st->bindValue(':coupon_id', $couponId, PDO::PARAM_INT);
            $st->bindValue(':deal_id', $dealId, PDO::PARAM_INT);
            $st->bindValue(':user_id', $userId, PDO::PARAM_INT);
            $st->bindValue(':used_by', $usedBy !== '' ? $usedBy : null, PDO::PARAM_STR);
            $st->execute();
            $response['logged_in_db'] = true;
        } else {
            $response['logged_in_db'] = false;
            $response['db_reason'] = 'coupon_not_found';
        }
    } catch (Throwable $e) {
        // Falha silenciosa para não quebrar o fluxo do painel
        $response['logged_in_db'] = false;
        $response['db_error'] = $e->getMessage();
    }
}

// Log leve em arquivo
try {
    $baseDir = __DIR__ . '/../storage/coupons_used';
    if (!is_dir($baseDir)) { @mkdir($baseDir, 0775, true); }

    $logData = [
        'type' => 'used',
        'ts' => date('c'),
        'deal_id' => $dealId,
        'coupon_code' => $couponCode,
        'coupon_title' => $couponTitle,
        'used_by' => $usedBy,
        'user_id' => (int) ($_SESSION['user_id'] ?? 0),
        'ip' => (string) ($_SERVER['REMOTE_ADDR'] ?? ''),
        'ua' => (string) ($_SERVER['HTTP_USER_AGENT'] ?? ''),
    ];

    $logFile = $baseDir . '/' . date('Y-m-d') . '.jsonl';
    @file_put_contents($logFile, json_encode($logData, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . "\n", FILE_APPEND);
} catch (Throwable $e) {
    // silencioso
}

// Notificação WhatsApp (opcional)
try {
    if (function_exists('nd_whatsapp_notify_merchant_coupon_used')) {
        $res = nd_whatsapp_notify_merchant_coupon_used($connect, $dealId, $couponCode, $couponTitle, $usedBy);
        if (!empty($res['ok']) || !empty($res['success'])) {
            $response['notified'] = true;
        }
    }
} catch (Throwable $e) {
    // silencioso
}

$response['ok'] = true;

echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
