<?php

declare(strict_types=1);
/**
 * guard_functions.php
 * - Protege TODAS as funções declaradas em:
 *     - functions.php (raiz)
 *     - admin/functions.php
 *   envolvendo cada uma em: if (!function_exists('nome')) { function nome(...) { ... } }
 * - Cria backup com timestamp antes de salvar.
 * - Idempotente (não duplica guarda se já existir).
 */

header('Content-Type: text/plain; charset=utf-8');

$targets = [
  __DIR__ . '/functions.php',
  __DIR__ . '/admin/functions.php',
];

function match_function_block($code, $bracePos) {
    $n = strlen($code);
    $j = $bracePos;
    $depth = 0;
    $inStr = null;
    $inSL = false; // // comment
    $inML = false; // /* */ comment

    while ($j < $n) {
        $ch  = $code[$j];
        $ch2 = ($j + 1 < $n) ? $code[$j] . $code[$j + 1] : '';

        if ($inStr !== null) {
            if ($ch === '\\') { $j += 2; continue; }
            if ($ch === $inStr) { $inStr = null; $j++; continue; }
            $j++; continue;
        }
        if ($inSL) {
            if ($ch === "\n") { $inSL = false; }
            $j++; continue;
        }
        if ($inML) {
            if ($ch2 === '*/') { $inML = false; $j += 2; continue; }
            $j++; continue;
        }

        if ($ch2 === '//') { $inSL = true; $j += 2; continue; }
        if ($ch2 === '/*') { $inML = true; $j += 2; continue; }
        if ($ch === '"' || $ch === "'") { $inStr = $ch; $j++; continue; }

        if ($ch === '{') { $depth++; $j++; continue; }
        if ($ch === '}') {
            $depth--;
            $j++;
            if ($depth === 0) {
                // retorna índice do primeiro char APÓS o bloco
                return $j;
            }
            continue;
        }
        $j++;
    }
    return false; // não encontrou fechamento
}

function wrap_all_functions_in_file($file) {
    if (!file_exists($file)) return ["MISS", 0, []];

    $code = file_get_contents($file);
    if ($code === false) return ["READ_ERR", 0, []];

    $tokens  = token_get_all($code);
    $offsets = [];
    $textByToken = [];
    $offset = 0;

    foreach ($tokens as $i => $tok) {
        if (is_array($tok)) {
            $txt = $tok[1];
        } else {
            $txt = $tok;
        }
        $textByToken[$i] = $txt;
        $offsets[$i] = $offset;
        $offset += strlen($txt);
    }

    $replacements = []; // [start, end, replacement]
    $wrappedNames = [];

    $countTokens = count($tokens);
    for ($i = 0; $i < $countTokens; $i++) {
        $tok = $tokens[$i];

        if (is_array($tok) && $tok[0] === T_FUNCTION) {
            // pula espaços e '&'
            $j = $i + 1;
            while ($j < $countTokens) {
                $t = $tokens[$j];
                if (is_array($t) && $t[0] === T_WHITESPACE) { $j++; continue; }
                if ($t === '&') { $j++; continue; }
                break;
            }
            // agora deve vir T_STRING com o nome
            if ($j < $countTokens && is_array($tokens[$j]) && $tokens[$j][0] === T_STRING) {
                $name = $tokens[$j][1];

                // posição inicial da palavra "function"
                $funcStart = $offsets[$i];

                // achar a primeira '{' depois da assinatura
                // avançar até encontrar '{'
                $k = $j + 1;
                $bracePos = null;
                while ($k < $countTokens) {
                    $tt = $tokens[$k];
                    $txt = is_array($tt) ? $tt[1] : $tt;
                    if ($txt === '{') {
                        $bracePos = $offsets[$k];
                        break;
                    }
                    $k++;
                }
                if ($bracePos === null) {
                    // função sem corpo padrão? ignora
                    continue;
                }

                // match do bloco da função até '}' correspondente
                $endPos = match_function_block($code, $bracePos);
                if ($endPos === false) {
                    // código desbalanceado, evita mexer
                    continue;
                }

                $funcBlock = substr($code, $funcStart, $endPos - $funcStart);

                // já está guardado?
                $lookbackStart = max(0, $funcStart - 400);
                $pre = substr($code, $lookbackStart, $funcStart - $lookbackStart);
                $guardRegex = "/if\\s*\\(\\s*!\\s*function_exists\\s*\\(\\s*['\"]" . preg_quote($name, '/') . "['\"]\\s*\\)\\s*\\)\\s*\\{/i";
                if (preg_match($guardRegex, $pre)) {
                    // já guardado — ignora
                    continue;
                }

                $wrapped = "if (!function_exists('".$name."')) {\n".$funcBlock."\n}\n";
                $replacements[] = [$funcStart, $endPos, $wrapped];
                $wrappedNames[] = $name;

                // pular tokens até perto do fim da função, para evitar reprocessar dentro
                // (não é estritamente necessário porque usamos offsets para aplicar)
            }
        }
    }

    if (empty($replacements)) {
        return ["NO_CHANGE", 0, []];
    }

    // aplica do fim pro começo para não deslocar offsets
    usort($replacements, function($a,$b){ return $b[0] <=> $a[0]; });

    $new = $code;
    foreach ($replacements as [$start, $end, $rep]) {
        $new = substr($new, 0, $start) . $rep . substr($new, $end);
    }

    // backup
    $bak = $file . '.bak.' . date('Ymd_His');
    @copy($file, $bak);
    file_put_contents($file, $new);

    return ["OK", count($replacements), $wrappedNames, $bak];
}

foreach ($targets as $t) {
    [$status, $cnt, $names, $bak] = wrap_all_functions_in_file($t);
    echo basename($t) . ": " . $status . " (wrapped: $cnt)\n";
    if (!empty($bak)) echo "Backup: $bak\n";
    if (!empty($names)) {
        foreach ($names as $nm) echo " - $nm\n";
    }
    echo "\n";
}

echo "Pronto. Agora rode flush_opcache.php e recarregue /admin.\n";
