<?php
declare(strict_types=1);

/**
 * Página pública: Ideias de campanha (para compartilhar no WhatsApp)
 * URL: /ideias-de-campanha
 *
 * Mostra ofertas reais do banco (deals) como inspiração de campanhas.
 */

require_once __DIR__ . '/core.php';
require_once __DIR__ . '/admin/functions.php'; // cleardata(), connect()

if (session_status() !== PHP_SESSION_ACTIVE) {
    @session_start();
}

// SEO
$titleSeoHeader = 'Ideias de campanha | ' . ($translation['tr_1'] ?? 'Conecta');
$descriptionSeoHeader = 'Exemplos reais de ofertas (do Conecta) para você copiar e publicar — com CTA e formato pronto para comércio local.';

/**
 * Busca ofertas reais (deals) ativas e públicas.
 */
function nd_get_campaign_ideas(PDO $db, int $limit = 12): array
{
    $now = function_exists('getDateByTimeZone') ? getDateByTimeZone() : date('Y-m-d H:i:s');

    $sql = "
        SELECT
            d.deal_id,
            d.deal_title,
            d.deal_slug,
            d.deal_image,
            d.deal_price,
            d.deal_oldprice,
            d.deal_category,
            d.deal_subcategory,
            d.deal_store,
            d.deal_location,
            d.deal_start,
            d.deal_expire,
            d.deal_featured,
            d.deal_exclusive,
            d.deal_created,
            c.category_title,
            s.store_title,
            (SELECT dc.coupon_code FROM deal_coupons dc WHERE dc.deal_id = d.deal_id AND dc.status = 1 ORDER BY dc.created_at DESC LIMIT 1) AS coupon_code,
            (SELECT dc.discount_percent FROM deal_coupons dc WHERE dc.deal_id = d.deal_id AND dc.status = 1 ORDER BY dc.created_at DESC LIMIT 1) AS discount_percent
        FROM deals d
        LEFT JOIN categories c ON d.deal_category = c.category_id
        LEFT JOIN stores s ON d.deal_store = s.store_id
        WHERE d.deal_status = 1
          AND d.deal_private = 0
          AND (CASE
                WHEN CAST(d.deal_start AS CHAR(25)) IN ('', '0000-00-00 00:00:00')
                     OR CAST(d.deal_start AS CHAR(10)) = '0000-00-00'
                     OR d.deal_start IS NULL
               THEN 1
               ELSE (d.deal_start <= :now)
              END)
          AND (CASE
                WHEN CAST(d.deal_expire AS CHAR(25)) IN ('', '0000-00-00 00:00:00')
                     OR CAST(d.deal_expire AS CHAR(10)) = '0000-00-00'
                     OR d.deal_expire IS NULL
               THEN 1
               ELSE (:now <= d.deal_expire)
              END)
        ORDER BY d.deal_featured DESC, d.deal_exclusive DESC, d.deal_created DESC
        LIMIT {$limit}
    ";

    $stmt = $db->prepare($sql);
    $stmt->execute([':now' => $now]);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    return is_array($items) ? $items : [];
}

// Carrega dados
$campaignDeals = [];
try {
    $db = $connect instanceof PDO ? $connect : connect();
    if ($db instanceof PDO) {
        $campaignDeals = nd_get_campaign_ideas($db, 12);
    }
} catch (Throwable $e) {
    $campaignDeals = [];
}

include __DIR__ . '/header.php';
include __DIR__ . '/sections/header.php';
require __DIR__ . '/views/campaign-ideas.view.php';
include __DIR__ . '/sections/footer.php';
include __DIR__ . '/footer.php';
