<?php
declare(strict_types=1);

/**
 * Pré-filtro de localização por IP (server-side)
 * Requer:
 *   - vendor/autoload.php (geoip2/geoip2)
 *   - storage/GeoLite2-City.mmdb
 * Efeito:
 *   - Se não houver cookie nd_location_slug, tenta resolver por IP e redireciona para /search?location=slug
 */

if (PHP_SAPI === 'cli') { return; }

/** Detecta IP real do cliente considerando CDN/proxy. */
function getClientIp(): string
{
    $s = $_SERVER;
    $candidates = [];
    if (!empty($s['HTTP_CF_CONNECTING_IP'])) { $candidates[] = $s['HTTP_CF_CONNECTING_IP']; }
    if (!empty($s['HTTP_X_FORWARDED_FOR']))  { $candidates[] = $s['HTTP_X_FORWARDED_FOR']; }
    if (!empty($s['HTTP_X_REAL_IP']))        { $candidates[] = $s['HTTP_X_REAL_IP']; }
    if (!empty($s['REMOTE_ADDR']))           { $candidates[] = $s['REMOTE_ADDR']; }

    foreach ($candidates as $raw) {
        $parts = preg_split('/\s*,\s*/', (string)$raw);
        foreach ($parts as $ip) {
            $ip = trim($ip);
            if ($ip !== '' && filter_var($ip, FILTER_VALIDATE_IP)) {
                return $ip;
            }
        }
    }
    return '';
}

/**
 * Obtém uma instância de PDO.
 * - Reaproveita conexões globais já existentes ($connect, $pdo, $db, $dbh).
 * - Se não houver, abre uma nova com as credenciais conhecidas.
 */
function getPdo(): ?\PDO
{
    foreach (['connect','pdo','db','dbh','database','DB'] as $name) {
        if (isset($GLOBALS[$name]) && $GLOBALS[$name] instanceof \PDO) {
            return $GLOBALS[$name];
        }
    }

    $host = defined('DB_HOST') ? DB_HOST : 'localhost';
    $name = defined('DB_NAME') ? DB_NAME : 'ndmidia_cupons';
    $user = defined('DB_USER') ? DB_USER : 'ndmidia_cupons';
    $pass = defined('DB_PASS') ? DB_PASS : '^)E[Z!Z][V=RCcEY';

    try {
        $dsn = "mysql:host={$host};dbname={$name};charset=utf8mb4";
        $pdo = new \PDO($dsn, $user, $pass, [
            \PDO::ATTR_ERRMODE            => \PDO::ERRMODE_EXCEPTION,
            \PDO::ATTR_DEFAULT_FETCH_MODE => \PDO::FETCH_ASSOC,
            // Emular prepares evita HY093 ao repetir nomes de parâmetros; ainda assim usaremos placeholders distintos
            \PDO::ATTR_EMULATE_PREPARES   => true,
        ]);
        return $pdo;
    } catch (\Throwable $e) {
        @file_put_contents(__DIR__.'/../storage/nearest.log', date('c').' pdo_error '.$e->getMessage()."\n", FILE_APPEND);
        return null;
    }
}

if (empty($_COOKIE['nd_location_slug'])) {
    $autoload = __DIR__ . '/../vendor/autoload.php';
    $mmdb     = __DIR__ . '/../storage/GeoLite2-City.mmdb';

    if (is_file($autoload) && is_file($mmdb)) {
        require_once $autoload;

        try {
            $ip = getClientIp();
            @file_put_contents(__DIR__.'/../storage/nearest.log', date('c')." hit ip=".$ip."\n", FILE_APPEND);

            if ($ip !== '') {
                $reader = new \GeoIp2\Database\Reader($mmdb);
                $rec    = $reader->city($ip);

                $lat = (float)($rec->location->latitude ?? 0);
                $lng = (float)($rec->location->longitude ?? 0);

                if ($lat && $lng) {
                    @file_put_contents(__DIR__.'/../storage/nearest.log', date('c')." geo lat=".$lat." lng=".$lng."\n", FILE_APPEND);

                    $pdo = getPdo();
                    if (!$pdo) {
                        @file_put_contents(__DIR__.'/../storage/nearest.log', date('c')." error no_pdo\n", FILE_APPEND);
                        return;
                    }

                    // Haversine com placeholders distintos (nada de :lat repetido)
                    $sql = "
                        SELECT location_title AS title, location_slug AS slug,
                               (6371 * ACOS(
                                 COS(RADIANS(:lat1))*COS(RADIANS(location_lat))*COS(RADIANS(location_lng)-RADIANS(:lng1)) +
                                 SIN(RADIANS(:lat2))*SIN(RADIANS(location_lat))
                               )) AS d
                        FROM locations
                        WHERE location_lat IS NOT NULL AND location_lng IS NOT NULL
                        ORDER BY d ASC
                        LIMIT 1";

                    $st = $pdo->prepare($sql);
                    $st->execute([
                        ':lat1' => $lat,
                        ':lat2' => $lat,
                        ':lng1' => $lng,
                    ]);

                    if ($row = $st->fetch(\PDO::FETCH_ASSOC)) {
                        $slug  = (string)($row['slug']  ?? '');
                        $title = (string)($row['title'] ?? '');

                        if ($slug !== '') {
                            $maxAge  = 365*24*60*60;
                            $isHttps = !empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off';
                            setcookie('nd_location_slug',  $slug,  time()+$maxAge, '/', '', $isHttps, true);
                            if ($title !== '') {
                                setcookie('nd_location_title', $title, time()+$maxAge, '/', '', $isHttps, true);
                            }

                            $uri = $_SERVER['REQUEST_URI'] ?? '/';
                            if (strpos($uri, '/search') !== 0) {
                                @file_put_contents(__DIR__.'/../storage/nearest.log', date('c')." redirect slug=".$slug." title=".$title."\n", FILE_APPEND);
                                header('Location: /search?location='.rawurlencode($slug), true, 302);
                                exit;
                            }
                        }
                    }
                }
            }
        } catch (\Throwable $e) {
            @file_put_contents(__DIR__.'/../storage/nearest.log', date('c').' error '.$e->getMessage()."\n", FILE_APPEND);
        }
    }
}
