<?php
declare(strict_types=1);

// ND Events: registro simples de eventos para relatórios + atribuição.
// Falha silenciosa se a tabela `nd_events` não existir.

if (!function_exists('nd_get_assoc_id')) {
  function nd_get_assoc_id(): ?int {
    $raw = $_COOKIE['nd_assoc_id'] ?? '';
    if ($raw === '' || $raw === null) return null;
    $id = (int)$raw;
    return ($id > 0) ? $id : null;
  }
}

if (!function_exists('nd_get_session_id')) {
  function nd_get_session_id(): string {
    $sid = $_COOKIE['nd_sid'] ?? '';
    if (is_string($sid) && preg_match('/^[a-f0-9]{24,64}$/i', $sid)) {
      return strtolower($sid);
    }

    // gera novo
    try {
      $sid = bin2hex(random_bytes(16));
    } catch (\Throwable $e) {
      $sid = sha1(uniqid('nd', true));
    }

    $isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
            || ((int)($_SERVER['SERVER_PORT'] ?? 0) === 443);

    $opts = [
      'expires'  => time() + (30 * 24 * 60 * 60),
      'path'     => '/',
      'secure'   => $isHttps,
      'httponly' => true,
      'samesite' => 'Lax',
    ];
    @setcookie('nd_sid', (string)$sid, $opts);

    return (string)$sid;
  }
}

if (!function_exists('nd_events_table_exists')) {
  function nd_events_table_exists(\PDO $db): bool {
    static $cached = null;
    if ($cached !== null) return (bool)$cached;

    try {
      $st = $db->query("SHOW TABLES LIKE 'nd_events'");
      $row = $st ? $st->fetch(\PDO::FETCH_NUM) : false;
      $cached = $row ? true : false;
    } catch (\Throwable $e) {
      $cached = false;
    }
    return (bool)$cached;
  }
}

if (!function_exists('nd_track_event')) {
  /**
   * @param PDO $db
   * @param string $event  Ex: view_deal, click_deal_link, copy_coupon
   * @param array $data    deal_id, store_id, coupon_code, meta (array)
   */
  function nd_track_event(\PDO $db, string $event, array $data = []): void {
    $event = strtolower(trim($event));
    if ($event === '' || !preg_match('/^[a-z0-9_]{1,32}$/', $event)) return;

    // evita gravar se a tabela não existir
    if (!nd_events_table_exists($db)) return;

    $dealId  = isset($data['deal_id'])  ? (int)$data['deal_id']  : null;
    $storeId = isset($data['store_id']) ? (int)$data['store_id'] : null;

    if ($dealId !== null && $dealId <= 0)   $dealId = null;
    if ($storeId !== null && $storeId <= 0) $storeId = null;

    $associateId = nd_get_assoc_id();
    $sid         = nd_get_session_id();

    // user_id (se logado)
    $userId = null;
    try {
      if (function_exists('getUserInfo')) {
        $u = (array)getUserInfo();
        if (!empty($u['user_id'])) $userId = (int)$u['user_id'];
      } elseif (!empty($_SESSION['user_id'])) {
        $userId = (int)$_SESSION['user_id'];
      }
    } catch (\Throwable $e) {}

    $ua   = (string)($_SERVER['HTTP_USER_AGENT'] ?? '');
    $ua   = mb_substr($ua, 0, 255, 'UTF-8');

    $ref  = (string)($_SERVER['HTTP_REFERER'] ?? '');
    $ref  = mb_substr($ref, 0, 255, 'UTF-8');

    $path = (string)($_SERVER['REQUEST_URI'] ?? '');
    $path = mb_substr($path, 0, 255, 'UTF-8');

    $ipRaw = (string)($_SERVER['HTTP_CF_CONNECTING_IP'] ?? ($_SERVER['HTTP_X_REAL_IP'] ?? ($_SERVER['REMOTE_ADDR'] ?? '')));
    $ipBin = null;
    if ($ipRaw !== '') {
      $packed = @inet_pton($ipRaw);
      if ($packed !== false) $ipBin = $packed; // VARBINARY(16)
    }

    $couponCode = isset($data['coupon_code']) ? (string)$data['coupon_code'] : null;
    if ($couponCode !== null) {
      $couponCode = trim($couponCode);
      if ($couponCode === '') $couponCode = null;
      if ($couponCode !== null) $couponCode = mb_substr($couponCode, 0, 64, 'UTF-8');
    }

    $meta = null;
    if (isset($data['meta']) && is_array($data['meta'])) {
      try {
        $meta = json_encode($data['meta'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
      } catch (\Throwable $e) {
        $meta = null;
      }
      if (is_string($meta) && strlen($meta) > 2000) {
        $meta = substr($meta, 0, 2000);
      }
    }

    try {
      $sql = "INSERT INTO nd_events
        (event_name, deal_id, store_id, associate_id, user_id, session_id, ip_bin, user_agent, referrer, path, coupon_code, meta_json, created_at)
        VALUES
        (:event_name, :deal_id, :store_id, :associate_id, :user_id, :session_id, :ip_bin, :user_agent, :referrer, :path, :coupon_code, :meta_json, NOW())";
      $st = $db->prepare($sql);
      $st->bindValue(':event_name', $event, \PDO::PARAM_STR);

      if ($dealId === null)  $st->bindValue(':deal_id', null, \PDO::PARAM_NULL);
      else                   $st->bindValue(':deal_id', $dealId, \PDO::PARAM_INT);

      if ($storeId === null) $st->bindValue(':store_id', null, \PDO::PARAM_NULL);
      else                   $st->bindValue(':store_id', $storeId, \PDO::PARAM_INT);

      if ($associateId === null) $st->bindValue(':associate_id', null, \PDO::PARAM_NULL);
      else                       $st->bindValue(':associate_id', $associateId, \PDO::PARAM_INT);

      if ($userId === null) $st->bindValue(':user_id', null, \PDO::PARAM_NULL);
      else                  $st->bindValue(':user_id', $userId, \PDO::PARAM_INT);

      $st->bindValue(':session_id', (string)$sid, \PDO::PARAM_STR);

      if ($ipBin === null) $st->bindValue(':ip_bin', null, \PDO::PARAM_NULL);
      else                 $st->bindValue(':ip_bin', $ipBin, \PDO::PARAM_LOB);

      $st->bindValue(':user_agent', $ua, \PDO::PARAM_STR);
      $st->bindValue(':referrer', $ref, \PDO::PARAM_STR);
      $st->bindValue(':path', $path, \PDO::PARAM_STR);

      if ($couponCode === null) $st->bindValue(':coupon_code', null, \PDO::PARAM_NULL);
      else                      $st->bindValue(':coupon_code', $couponCode, \PDO::PARAM_STR);

      if ($meta === null) $st->bindValue(':meta_json', null, \PDO::PARAM_NULL);
      else                $st->bindValue(':meta_json', $meta, \PDO::PARAM_STR);

      $st->execute();
    } catch (\Throwable $e) {
      // silencioso
      return;
    }
  }
}
