<?php
declare(strict_types=1);

/**
 * Public Report (per store token)
 * URL: /public-report.php?t=<token>
 * - No login required
 * - Shows key KPIs and top offers for a single store
 */

if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }

require_once __DIR__ . '/core.php';

// DB connect (defensive)
$connect = $connect ?? null;
try {
  if (!$connect && function_exists('connect')) {
    if (isset($database)) { $connect = @connect($database); }
    if (!$connect)        { $connect = @connect(); }
  }
} catch (Throwable $e) {
  $connect = null;
}

// Basic helpers
if (!function_exists('nd_h')) {
  function nd_h($v): string {
    return htmlspecialchars((string)$v, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
  }
}

$token = isset($_GET['t']) ? trim((string)$_GET['t']) : '';
$token = preg_replace('/[^a-f0-9]/i', '', $token);
if ($token === '' || strlen($token) < 24 || strlen($token) > 64) {
  http_response_code(404);
  echo 'Relatório não encontrado.';
  exit;
}

// Check required tables
$hasTokens = false;
$hasEvents  = false;
try {
  if ($connect) {
    $q = $connect->query("SHOW TABLES LIKE 'nd_report_tokens'");
    $hasTokens = (bool)($q && $q->fetch(PDO::FETCH_NUM));
    $q2 = $connect->query("SHOW TABLES LIKE 'nd_events'");
    $hasEvents = (bool)($q2 && $q2->fetch(PDO::FETCH_NUM));
  }
} catch (Throwable $e) {}

if (!$connect || !$hasTokens) {
  http_response_code(404);
  echo 'Relatório indisponível.';
  exit;
}

// Resolve token -> store
$store = null;
$storeId = 0;
$userId  = 0;
try {
  $st = $connect->prepare(
    "SELECT t.store_id, t.user_id, s.store_title, s.store_image, s.store_whatsapp, s.store_url, s.store_address\n".
    "FROM nd_report_tokens t\n".
    "LEFT JOIN stores s ON s.store_id = t.store_id\n".
    "WHERE t.token = :t AND t.is_revoked = 0\n".
    "LIMIT 1"
  );
  $st->execute([':t' => $token]);
  $store = $st->fetch(PDO::FETCH_ASSOC) ?: null;
  $storeId = (int)($store['store_id'] ?? 0);
  $userId  = (int)($store['user_id'] ?? 0);
} catch (Throwable $e) {
  $store = null;
}

if (!$store || $storeId <= 0) {
  http_response_code(404);
  echo 'Relatório não encontrado.';
  exit;
}

// Touch last_used_at (best-effort)
try {
  $u = $connect->prepare("UPDATE nd_report_tokens SET last_used_at = NOW() WHERE token = :t");
  $u->execute([':t' => $token]);
} catch (Throwable $e) {}

// Date range (default last 30 days)
$tz = new DateTimeZone('America/Sao_Paulo');
$today = new DateTime('now', $tz);

$fromStr = isset($_GET['from']) ? (string)$_GET['from'] : '';
$toStr   = isset($_GET['to'])   ? (string)$_GET['to']   : '';

$start = null; $end = null;
try {
  if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $fromStr)) $start = new DateTime($fromStr . ' 00:00:00', $tz);
  if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $toStr))   $end   = new DateTime($toStr   . ' 23:59:59', $tz);
} catch (Throwable $e) {}

if (!$start) { $start = (clone $today)->modify('-30 days')->setTime(0,0,0); }
if (!$end)   { $end   = (clone $today)->setTime(23,59,59); }

// KPIs
$summary = ['views'=>0,'clicks'=>0,'copies'=>0,'sessions'=>0];
$topDeals = [];

if ($hasEvents) {
  try {
    $sql = "SELECT\n".
           "  SUM(event_name = 'view_deal') AS views,\n".
           "  SUM(event_name = 'click_deal_link') AS clicks,\n".
           "  SUM(event_name = 'copy_coupon') AS copies,\n".
           "  COUNT(DISTINCT session_id) AS sessions\n".
           "FROM nd_events\n".
           "WHERE store_id = :sid AND created_at BETWEEN :a AND :b";
    $st = $connect->prepare($sql);
    $st->execute([
      ':sid' => $storeId,
      ':a'   => $start->format('Y-m-d H:i:s'),
      ':b'   => $end->format('Y-m-d H:i:s'),
    ]);
    $row = $st->fetch(PDO::FETCH_ASSOC) ?: [];
    $summary = [
      'views'    => (int)($row['views'] ?? 0),
      'clicks'   => (int)($row['clicks'] ?? 0),
      'copies'   => (int)($row['copies'] ?? 0),
      'sessions' => (int)($row['sessions'] ?? 0),
    ];
  } catch (Throwable $e) {
    $summary = ['views'=>0,'clicks'=>0,'copies'=>0,'sessions'=>0];
  }

  try {
    $sql = "SELECT d.deal_id, d.deal_title,\n".
           "  SUM(e.event_name = 'view_deal') AS views,\n".
           "  SUM(e.event_name = 'click_deal_link') AS clicks,\n".
           "  SUM(e.event_name = 'copy_coupon') AS coupon_copies\n".
           "FROM nd_events e\n".
           "LEFT JOIN deals d ON d.deal_id = e.deal_id\n".
           "WHERE e.store_id = :sid AND e.created_at BETWEEN :a AND :b\n".
           "GROUP BY d.deal_id, d.deal_title\n".
           "ORDER BY clicks DESC, views DESC\n".
           "LIMIT 10";
    $st = $connect->prepare($sql);
    $st->execute([
      ':sid' => $storeId,
      ':a'   => $start->format('Y-m-d H:i:s'),
      ':b'   => $end->format('Y-m-d H:i:s'),
    ]);
    $topDeals = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  } catch (Throwable $e) {
    $topDeals = [];
  }
}

// Month-to-date headline
$mtd = ['views'=>0,'clicks'=>0,'copies'=>0];
if ($hasEvents) {
  try {
    $monthStart = new DateTime('first day of this month 00:00:00', $tz);
    $sql = "SELECT\n".
           "  SUM(event_name = 'view_deal') AS views,\n".
           "  SUM(event_name = 'click_deal_link') AS clicks,\n".
           "  SUM(event_name = 'copy_coupon') AS copies\n".
           "FROM nd_events\n".
           "WHERE store_id = :sid AND created_at BETWEEN :a AND :b";
    $st = $connect->prepare($sql);
    $st->execute([
      ':sid' => $storeId,
      ':a'   => $monthStart->format('Y-m-d H:i:s'),
      ':b'   => $today->format('Y-m-d H:i:s'),
    ]);
    $row = $st->fetch(PDO::FETCH_ASSOC) ?: [];
    $mtd = [
      'views'  => (int)($row['views'] ?? 0),
      'clicks' => (int)($row['clicks'] ?? 0),
      'copies' => (int)($row['copies'] ?? 0),
    ];
  } catch (Throwable $e) {
    $mtd = ['views'=>0,'clicks'=>0,'copies'=>0];
  }
}

$views  = (int)($summary['views'] ?? 0);
$clicks = (int)($summary['clicks'] ?? 0);
$copies = (int)($summary['copies'] ?? 0);
$ctr = ($views > 0) ? round(($clicks / $views) * 100, 1) : 0.0;

function nd_pct(float $v): string {
  return number_format($v, 1, ',', '.') . '%';
}

?><!doctype html>
<html lang="pt-BR">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title><?php echo nd_h($store['store_title'] ?? 'Relatório'); ?> • Relatório</title>
  <link rel="stylesheet" href="/assets/css/uikit.css">
  <link rel="stylesheet" href="/assets/css/styles.css">
  <link rel="stylesheet" href="/assets/css/offer-pro.css">
  <link rel="stylesheet" href="/assets/css/nd-ui-refresh.css?v=1">
  <script src="/assets/js/jquery.js"></script>
  <script src="/assets/js/uikit.js"></script>
  <script src="/assets/js/uikit-icons.js"></script>
</head>
<body class="uk-background-muted">

<div class="uk-section-small">
  <div class="uk-container uk-container-large">

    <div class="uk-flex uk-flex-middle uk-flex-between" uk-grid>
      <div class="uk-width-expand@m">
        <div class="uk-flex uk-flex-middle" uk-grid>
          <div class="uk-width-auto">
            <?php if (!empty($store['store_image'])): ?>
              <img src="/images/stores/<?php echo nd_h($store['store_image']); ?>" alt="" style="width:52px;height:52px;border-radius:12px;object-fit:cover;">
            <?php else: ?>
              <div class="uk-border-rounded uk-background-default" style="width:52px;height:52px;"></div>
            <?php endif; ?>
          </div>
          <div class="uk-width-expand">
            <div class="uk-text-small uk-text-muted">Relatório de desempenho</div>
            <h2 class="uk-margin-remove"><?php echo nd_h($store['store_title'] ?? 'Loja'); ?></h2>
          </div>
        </div>
      </div>
      <div class="uk-width-auto@m uk-visible@m">
        <a class="uk-button uk-button-default uk-border-rounded" href="/">Voltar ao site</a>
      </div>
    </div>

    <div class="uk-card uk-card-primary uk-card-body uk-border-rounded uk-margin-top" style="padding:16px 18px;">
      <div class="uk-text-bold" style="font-size:16px;">
        Este mês: <span style="font-size:18px;"><?php echo (int)$mtd['clicks']; ?></span> cliques no CTA
      </div>
      <div class="uk-text-small" style="opacity:.9;">
        <?php echo (int)$mtd['views']; ?> visualizações • <?php echo (int)$mtd['copies']; ?> cupons copiados
      </div>
    </div>

    <div class="uk-card uk-card-default uk-card-body uk-border-rounded uk-margin-top">
      <form class="uk-grid-small" uk-grid method="get" action="">
        <input type="hidden" name="t" value="<?php echo nd_h($token); ?>" />
        <div class="uk-width-1-2@m">
          <label class="uk-form-label">De</label>
          <input class="uk-input" type="date" name="from" value="<?php echo nd_h($start->format('Y-m-d')); ?>" />
        </div>
        <div class="uk-width-1-2@m">
          <label class="uk-form-label">Até</label>
          <input class="uk-input" type="date" name="to" value="<?php echo nd_h($end->format('Y-m-d')); ?>" />
        </div>
        <div class="uk-width-1-1 uk-margin-small-top">
          <button class="uk-button uk-button-primary uk-border-rounded" type="submit">Filtrar</button>
        </div>
      </form>
    </div>

    <?php if (!$hasEvents): ?>
      <div class="uk-alert-warning uk-margin-top" uk-alert>
        <p class="uk-margin-remove">Relatórios ainda não foram ativados (tabela <code>nd_events</code> ausente).</p>
      </div>
    <?php endif; ?>

    <div class="uk-grid-small uk-child-width-1-1 uk-child-width-1-2@m uk-child-width-1-4@l uk-margin-top" uk-grid>
      <div>
        <div class="uk-card uk-card-default uk-card-body uk-border-rounded">
          <div class="uk-text-meta">Visualizações</div>
          <div class="uk-text-large uk-text-bold"><?php echo (int)$summary['views']; ?></div>
          <div class="uk-text-meta">Ofertas abertas no período</div>
        </div>
      </div>
      <div>
        <div class="uk-card uk-card-default uk-card-body uk-border-rounded">
          <div class="uk-text-meta">Cliques no CTA</div>
          <div class="uk-text-large uk-text-bold"><?php echo (int)$summary['clicks']; ?></div>
          <div class="uk-text-meta">Cliques para visitar / comprar</div>
        </div>
      </div>
      <div>
        <div class="uk-card uk-card-default uk-card-body uk-border-rounded">
          <div class="uk-text-meta">CTR</div>
          <div class="uk-text-large uk-text-bold"><?php echo nd_pct((float)$ctr); ?></div>
          <div class="uk-text-meta">Cliques / visualizações</div>
        </div>
      </div>
      <div>
        <div class="uk-card uk-card-default uk-card-body uk-border-rounded">
          <div class="uk-text-meta">Cupons copiados</div>
          <div class="uk-text-large uk-text-bold"><?php echo (int)$summary['copies']; ?></div>
          <div class="uk-text-meta">Intenção de compra</div>
        </div>
      </div>
    </div>

    <div class="uk-card uk-card-default uk-card-body uk-border-rounded uk-margin-top">
      <h3 class="uk-margin-remove-bottom">Top ofertas (período)</h3>
      <div class="uk-text-meta uk-margin-small-top">Ordenado por cliques no CTA e visualizações.</div>
      <div class="uk-overflow-auto uk-margin-medium-top">
        <table class="uk-table uk-table-divider uk-table-middle">
          <thead>
            <tr>
              <th>Oferta</th>
              <th class="uk-text-right">Views</th>
              <th class="uk-text-right">Cliques</th>
              <th class="uk-text-right">CTR</th>
              <th class="uk-text-right">Cupom</th>
            </tr>
          </thead>
          <tbody>
            <?php if (empty($topDeals)): ?>
              <tr><td colspan="5" class="uk-text-center uk-text-meta">Sem dados no período selecionado.</td></tr>
            <?php else: ?>
              <?php foreach ($topDeals as $d): ?>
                <?php
                  $dv = (int)($d['views'] ?? 0);
                  $dc = (int)($d['clicks'] ?? 0);
                  $dctr = ($dv > 0) ? round(($dc / $dv) * 100, 1) : 0.0;
                ?>
                <tr>
                  <td>
                    <div class="uk-text-bold"><?php echo nd_h($d['deal_title'] ?? ('#' . (int)($d['deal_id'] ?? 0))); ?></div>
                    <div class="uk-text-meta">ID: <?php echo (int)($d['deal_id'] ?? 0); ?></div>
                  </td>
                  <td class="uk-text-right"><?php echo $dv; ?></td>
                  <td class="uk-text-right"><?php echo $dc; ?></td>
                  <td class="uk-text-right"><?php echo nd_pct((float)$dctr); ?></td>
                  <td class="uk-text-right"><?php echo (int)($d['coupon_copies'] ?? 0); ?></td>
                </tr>
              <?php endforeach; ?>
            <?php endif; ?>
          </tbody>
        </table>
      </div>
      <div class="uk-text-meta">* Dica: use este link para mostrar ROI em tempo real no WhatsApp.</div>
    </div>

  </div>
</div>

</body>
</html>
