<?php

declare(strict_types=1);
/*--------------------*/
// QR tracking para associados
// - Lê o código do associado (?code= ou ?a=)
// - Registra o clique na tabela associate_clicks
// - Garante no máximo 1 registro por mês por IP + associado
// - Redireciona para a página inicial do portal
/*--------------------*/

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/admin/functions.php';

/**
 * Descobre o IP real do visitante, mesmo atrás de proxy/Cloudflare.
 */
function nd_get_real_ip(): string
{
    $keys = array(
        'HTTP_CF_CONNECTING_IP',
        'HTTP_X_REAL_IP',
        'HTTP_X_FORWARDED_FOR',
        'HTTP_CLIENT_IP',
        'REMOTE_ADDR',
    );

    foreach ($keys as $key) {
        if (!empty($_SERVER[$key])) {
            $ipList = explode(',', (string)$_SERVER[$key]);
            $ip = trim($ipList[0]);
            if (filter_var($ip, FILTER_VALIDATE_IP)) {
                return $ip;
            }
        }
    }

    return '0.0.0.0';
}

// --- 1) Código do associado vindo do QR (?code= ou ?a=) ---
$code = isset($_GET['code']) ? trim((string)$_GET['code']) : '';
if ($code === '' && isset($_GET['a'])) {
    $code = trim((string)$_GET['a']);
}

// Se não tiver código, só redireciona
if ($code === '') {
    header('Location: ' . SITE_URL);
    exit;
}

try {
    $db = connect();

    if ($db instanceof PDO) {
        // Localiza o associado pelo código
        $stmt = $db->prepare(
            "SELECT associate_id 
             FROM associates 
             WHERE associate_code = :code 
             LIMIT 1"
        );
        $stmt->execute(array(':code' => $code));
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row && isset($row['associate_id'])) {
            $associateId = (int)$row['associate_id'];

            // 0) Persistência (atribuição): salva o associado em cookie por 30 dias
            // - Ajuda a atribuir futuras ações (abrir ofertas, clicar no WhatsApp, etc.) ao associado
            // - Não expõe dados sensíveis (apenas id e código)
            $isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
                      || ((int)($_SERVER['SERVER_PORT'] ?? 0) === 443);
            $cookieOpts = [
                'expires'  => time() + (30 * 24 * 60 * 60),
                'path'     => '/',
                'secure'   => $isHttps,
                'httponly' => true,
                'samesite' => 'Lax',
            ];
            @setcookie('nd_assoc_code', (string)$code, $cookieOpts);
            @setcookie('nd_assoc_id', (string)$associateId, $cookieOpts);

            // IP e User-Agent normalizados
            $ip = nd_get_real_ip();
            $ip = mb_substr($ip, 0, 45, 'UTF-8');

            $ua = isset($_SERVER['HTTP_USER_AGENT']) ? (string)$_SERVER['HTTP_USER_AGENT'] : '';
            $ua = mb_substr($ua, 0, 255, 'UTF-8');

            // 2) Verifica se já existe clique neste mês para esse associado + IP
            try {
                $checkSql = "
                    SELECT click_id
                    FROM associate_clicks
                    WHERE associate_id = :id
                      AND click_ip = :ip
                      AND YEAR(click_time) = YEAR(NOW())
                      AND MONTH(click_time) = MONTH(NOW())
                    LIMIT 1
                ";
                $checkStmt = $db->prepare($checkSql);
                $checkStmt->execute(array(
                    ':id' => $associateId,
                    ':ip' => $ip,
                ));
                $already = $checkStmt->fetch(PDO::FETCH_ASSOC);
            } catch (Exception $e) {
                // Se der erro na verificação, considera como se não existisse
                $already = false;
            }

            // 3) Só grava se ainda não tiver um registro neste mês para esse IP
            if (!$already) {
                try {
                    $insertSql = "
                        INSERT INTO associate_clicks (associate_id, click_time, click_ip, click_user_agent)
                        VALUES (:id, NOW(), :ip, :ua)
                    ";
                    $clickStmt = $db->prepare($insertSql);
                    $clickStmt->execute(array(
                        ':id' => $associateId,
                        ':ip' => $ip,
                        ':ua' => $ua,
                    ));
                } catch (Exception $e) {
                    // Se a tabela não existir ou der erro, ignoramos para não quebrar nada
                }
            }
        }
    }
} catch (Exception $e) {
    // Qualquer erro de conexão/consulta é silencioso para não quebrar o fluxo
}

// Redireciona para a home do site
header('Location: ' . SITE_URL);
exit;

