<?php
declare(strict_types=1);

/**
 * Landing + captação de anunciantes
 * URL sugerida: /anunciantes (via .htaccess)
 */

require_once __DIR__ . '/core.php';
require_once __DIR__ . '/admin/functions.php'; // cleardata(), connect()

if (session_status() !== PHP_SESSION_ACTIVE) {
    @session_start();
}

$errors  = array();
$success = '';

$company = '';
$name    = '';
$email   = '';
$phone   = '';
$city    = '';
$state   = '';
$segment = '';
$objective = '';
$message = '';

// SEO
$titleSeoHeader = 'Anuncie no Conecta | ' . ($translation['tr_1'] ?? 'Conecta');
$descriptionSeoHeader = 'Cadastre seu interesse e receba uma proposta para anunciar no ecossistema Conecta (comércio local + rede de associados).';

/**
 * Cria a tabela de leads se não existir (best-effort)
 */
function nd_ensure_advertiser_leads_table(PDO $db): void
{
    $sql = "CREATE TABLE IF NOT EXISTS advertiser_leads (
        lead_id INT UNSIGNED NOT NULL AUTO_INCREMENT,
        company_name VARCHAR(180) NOT NULL,
        contact_name VARCHAR(150) NOT NULL,
        contact_email VARCHAR(180) NOT NULL,
        contact_phone VARCHAR(60) DEFAULT NULL,
        city VARCHAR(120) DEFAULT NULL,
        state VARCHAR(10) DEFAULT NULL,
        segment VARCHAR(120) DEFAULT NULL,
        objective VARCHAR(120) DEFAULT NULL,
        message TEXT,
        lead_status TINYINT(1) NOT NULL DEFAULT 0,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (lead_id),
        KEY idx_created_at (created_at),
        KEY idx_email (contact_email)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci";
    try { $db->exec($sql); } catch (Throwable $e) { /* silencioso */ }
}

/**
 * Envia e-mail de notificação (best-effort)
 */
function nd_notify_admin_new_advertiser_lead(array $payload, array $settings): void
{
    if (!function_exists('sendMail')) return;
    if (empty($settings['st_recipientemail'])) return;

    $to = (string)$settings['st_recipientemail'];
    $fromName = 'Novo lead (Anunciante)';
    $subject  = '[Conecta] Novo lead anunciante: {COMPANY_NAME}';

    $emailHtml = '<div style="font-family:Arial,sans-serif;font-size:14px;line-height:1.45">'
        . '<h2 style="margin:0 0 12px">Novo interesse de anunciante</h2>'
        . '<p><strong>Empresa:</strong> {COMPANY_NAME}<br>'
        . '<strong>Contato:</strong> {CONTACT_NAME}<br>'
        . '<strong>E-mail:</strong> {CONTACT_EMAIL}<br>'
        . '<strong>WhatsApp:</strong> {CONTACT_PHONE}<br>'
        . '<strong>Cidade/UF:</strong> {CITY} / {STATE}<br>'
        . '<strong>Segmento:</strong> {SEGMENT}<br>'
        . '<strong>Objetivo:</strong> {OBJECTIVE}</p>'
        . '<p><strong>Mensagem:</strong><br>{MESSAGE}</p>'
        . '</div>';

    $vars = array(
        '{COMPANY_NAME}'  => $payload['company'] ?? '',
        '{CONTACT_NAME}'  => $payload['name'] ?? '',
        '{CONTACT_EMAIL}' => $payload['email'] ?? '',
        '{CONTACT_PHONE}' => $payload['phone'] ?? '',
        '{CITY}'          => $payload['city'] ?? '',
        '{STATE}'         => $payload['state'] ?? '',
        '{SEGMENT}'       => $payload['segment'] ?? '',
        '{OBJECTIVE}'     => $payload['objective'] ?? '',
        '{MESSAGE}'       => nl2br(htmlspecialchars((string)($payload['message'] ?? ''), ENT_QUOTES, 'UTF-8')),
    );

    try {
        sendMail($vars, $emailHtml, $to, $fromName, $subject, true, (string)($payload['name'] ?? ''), (string)($payload['email'] ?? ''));
    } catch (Throwable $e) {
        // silencioso
    }
}

// POST
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'POST') {
    $company   = cleardata($_POST['company_name'] ?? '');
    $name      = cleardata($_POST['contact_name'] ?? '');
    $email     = cleardata($_POST['contact_email'] ?? '');
    $phone     = cleardata($_POST['contact_phone'] ?? '');
    $city      = cleardata($_POST['city'] ?? '');
    $state     = strtoupper(trim((string)cleardata($_POST['state'] ?? '')));
    $segment   = cleardata($_POST['segment'] ?? '');
    $objective = cleardata($_POST['objective'] ?? '');
    $message   = cleardata($_POST['message'] ?? '');
    $consent   = isset($_POST['consent']) ? (string)$_POST['consent'] : '';

    if ($company === '') $errors[] = 'Informe o nome da empresa.';
    if ($name === '') $errors[] = 'Informe o nome do responsável.';
    if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = 'Informe um e-mail válido.';
    if ($state !== '' && strlen($state) !== 2) $errors[] = 'Informe a UF com 2 letras (ex: SC).';
    if ($consent !== '1') $errors[] = 'Você precisa aceitar o contato para prosseguirmos.';

    if (empty($errors)) {
        try {
            $db = $connect instanceof PDO ? $connect : connect();
            if ($db instanceof PDO) {
                nd_ensure_advertiser_leads_table($db);

                $stmt = $db->prepare(
                    'INSERT INTO advertiser_leads
                    (company_name, contact_name, contact_email, contact_phone, city, state, segment, objective, message, lead_status, created_at)
                    VALUES
                    (:company, :name, :email, :phone, :city, :state, :segment, :objective, :message, 0, NOW())'
                );
                $stmt->execute([
                    ':company'   => $company,
                    ':name'      => $name,
                    ':email'     => $email,
                    ':phone'     => $phone,
                    ':city'      => $city,
                    ':state'     => $state,
                    ':segment'   => $segment,
                    ':objective' => $objective,
                    ':message'   => $message,
                ]);
            }

            nd_notify_admin_new_advertiser_lead([
                'company' => $company,
                'name' => $name,
                'email' => $email,
                'phone' => $phone,
                'city' => $city,
                'state' => $state,
                'segment' => $segment,
                'objective' => $objective,
                'message' => $message,
            ], $settings);

            $success = 'Recebemos seu interesse! Em breve vamos te chamar no WhatsApp/E-mail com uma proposta.';

            // limpa
            $company = $name = $email = $phone = $city = $state = $segment = $objective = $message = '';

        } catch (Throwable $e) {
            $errors[] = 'Ocorreu um erro ao enviar. Tente novamente em instantes.';
        }
    }
}

include __DIR__ . '/header.php';
include __DIR__ . '/sections/header.php';
require __DIR__ . '/views/lead-advertiser.view.php';
include __DIR__ . '/sections/views/footer-ad.view.php';
include __DIR__ . '/sections/footer.php';
include __DIR__ . '/footer.php';
