<?php
declare(strict_types=1);

/**
 * Landing + cadastro de associados (motoristas / apps de transporte)
 * URL sugerida: /associados (via .htaccess)
 */

require_once __DIR__ . '/core.php';
require_once __DIR__ . '/admin/functions.php'; // cleardata(), connect()

if (session_status() !== PHP_SESSION_ACTIVE) {
    @session_start();
}

$errors  = array();
$success = '';

$name  = '';
$email = '';
$phone = '';
$city  = '';
$state = '';
$group = '';

// SEO
$titleSeoHeader = 'Seja um Associado | ' . ($translation['tr_1'] ?? 'Conecta');
$descriptionSeoHeader = 'Cadastre-se como associado (Uber, 99, Táxi e similares). Receba seu QR Code e acompanhe seus cliques no painel.';

/**
 * Envia e-mail de notificação (best-effort)
 */
function nd_notify_admin_new_associate(array $payload, array $settings): void
{
    if (!function_exists('sendMail')) return;
    if (empty($settings['st_recipientemail'])) return;

    $to = (string)$settings['st_recipientemail'];
    $fromName = 'Novo cadastro (Associado)';
    $subject  = '[Conecta] Novo cadastro de associado: {ASSOC_NAME}';

    $emailHtml = '<div style="font-family:Arial,sans-serif;font-size:14px;line-height:1.45">'
        . '<h2 style="margin:0 0 12px">Novo cadastro de associado</h2>'
        . '<p><strong>Nome:</strong> {ASSOC_NAME}<br>'
        . '<strong>E-mail:</strong> {ASSOC_EMAIL}<br>'
        . '<strong>WhatsApp:</strong> {ASSOC_PHONE}<br>'
        . '<strong>Cidade/UF:</strong> {ASSOC_CITY} / {ASSOC_STATE}<br>'
        . '<strong>Grupo:</strong> {ASSOC_GROUP}<br>'
        . '<strong>Código:</strong> <code>{ASSOC_CODE}</code></p>'
        . '<p style="color:#666">Status foi criado como <strong>pendente</strong>. Você pode ativar no painel admin.</p>'
        . '</div>';

    $vars = array(
        '{ASSOC_NAME}'  => $payload['name'] ?? '',
        '{ASSOC_EMAIL}' => $payload['email'] ?? '',
        '{ASSOC_PHONE}' => $payload['phone'] ?? '',
        '{ASSOC_CITY}'  => $payload['city'] ?? '',
        '{ASSOC_STATE}' => $payload['state'] ?? '',
        '{ASSOC_GROUP}' => $payload['group'] ?? '',
        '{ASSOC_CODE}'  => $payload['code'] ?? '',
    );

    try {
        // reply-to no e-mail do associado
        sendMail($vars, $emailHtml, $to, $fromName, $subject, true, (string)($payload['name'] ?? ''), (string)($payload['email'] ?? ''));
    } catch (Throwable $e) {
        // silencioso
    }
}

// POST
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'POST') {
    $name  = cleardata($_POST['associate_name']  ?? '');
    $email = cleardata($_POST['associate_email'] ?? '');
    $phone = cleardata($_POST['associate_phone'] ?? '');
    $city  = cleardata($_POST['associate_city']  ?? '');
    $state = strtoupper(trim((string)cleardata($_POST['associate_state'] ?? '')));
    $group = cleardata($_POST['associate_group'] ?? '');
    $consent = isset($_POST['consent']) ? (string)$_POST['consent'] : '';

    if ($name === '') $errors[] = 'Informe seu nome.';
    if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = 'Informe um e-mail válido.';
    if ($phone === '') $errors[] = 'Informe seu WhatsApp.';
    if ($city === '') $errors[] = 'Informe sua cidade.';
    if ($state !== '' && strlen($state) !== 2) $errors[] = 'Informe a UF com 2 letras (ex: SC).';
    if ($consent !== '1') $errors[] = 'Confirme o consentimento para contato.';

    // valida grupo
    $allowedGroups = array('Uber','99','Táxi','Cooperativa','Outros');
    if ($group === '' || !in_array($group, $allowedGroups, true)) {
        $group = 'Outros';
    }

    // e-mail duplicado
    if (empty($errors)) {
        try {
            $db = connect();
            $checkEmail = $db->prepare('SELECT 1 FROM associates WHERE associate_email = :email LIMIT 1');
            $checkEmail->execute([':email' => $email]);
            if ($checkEmail->fetch()) {
                $errors[] = 'Este e-mail já está cadastrado como associado.';
            }
        } catch (Throwable $e) {
            // ignora
        }
    }

    // gera código
    $code = '';
    if (empty($errors)) {
        try {
            $db = connect();
            $unique = false;
            $attempts = 0;
            while (!$unique && $attempts < 10) {
                $attempts++;
                $generated = 'A' . strtoupper(substr(sha1(uniqid((string)mt_rand(), true)), 0, 7));
                $check = $db->prepare('SELECT 1 FROM associates WHERE associate_code = :code LIMIT 1');
                $check->execute([':code' => $generated]);
                if (!$check->fetch()) {
                    $unique = true;
                    $code = $generated;
                }
            }
            if ($code === '') {
                $code = 'A' . strtoupper(substr(sha1((string)time()), 0, 7));
            }
        } catch (Throwable $e) {
            $code = 'A' . strtoupper(substr(sha1((string)time()), 0, 7));
        }
    }

    if (empty($errors)) {
        try {
            $db = connect();

            // compat: coluna associate_group pode não existir
            $hasGroup = false;
            try {
                $checkCol = $db->query("SHOW COLUMNS FROM associates LIKE 'associate_group'");
                if ($checkCol && $checkCol->rowCount() > 0) $hasGroup = true;
            } catch (Throwable $e) {
                $hasGroup = false;
            }

            $status = 0; // pendente (aprovação)
            if ($hasGroup) {
                $stmt = $db->prepare(
                    'INSERT INTO associates
                    (associate_name, associate_email, associate_phone, associate_city, associate_state, associate_group, associate_code, associate_status, associate_created)
                    VALUES
                    (:name, :email, :phone, :city, :state, :grupo, :code, :status, NOW())'
                );
                $stmt->execute([
                    ':name' => $name,
                    ':email' => $email,
                    ':phone' => $phone,
                    ':city' => $city,
                    ':state' => $state,
                    ':grupo' => $group,
                    ':code' => $code,
                    ':status' => $status,
                ]);
            } else {
                $stmt = $db->prepare(
                    'INSERT INTO associates
                    (associate_name, associate_email, associate_phone, associate_city, associate_state, associate_code, associate_status, associate_created)
                    VALUES
                    (:name, :email, :phone, :city, :state, :code, :status, NOW())'
                );
                $stmt->execute([
                    ':name' => $name,
                    ':email' => $email,
                    ':phone' => $phone,
                    ':city' => $city,
                    ':state' => $state,
                    ':code' => $code,
                    ':status' => $status,
                ]);
            }

            nd_notify_admin_new_associate([
                'name' => $name,
                'email' => $email,
                'phone' => $phone,
                'city' => $city,
                'state' => $state,
                'group' => $group,
                'code' => $code,
            ], $settings);

            $success = 'Cadastro recebido! Seu código de associado é: ' . $code;

            // limpa
            $name = $email = $phone = $city = $state = $group = '';

        } catch (Throwable $e) {
            $errors[] = 'Ocorreu um erro ao salvar seu cadastro. Tente novamente em instantes.';
        }
    }
}

include __DIR__ . '/header.php';
include __DIR__ . '/sections/header.php';
require __DIR__ . '/views/lead-associate.view.php';
include __DIR__ . '/sections/views/footer-ad.view.php';
include __DIR__ . '/sections/footer.php';
include __DIR__ . '/footer.php';
