<?php declare(strict_types=1);
/**
 * /signin.php — versão limpa e padronizada
 * - strict_types é a PRIMEIRA instrução
 * - Sem saída antes do PHP
 * - Apenas UM require de core.php
 * - Sem blocos duplicados de Google Translate (usa somente o partial)
 * - Sem "?>" final
 */

// Buffer defensivo, sem forçar output antes de redirects
if (function_exists('ob_get_level') && ob_get_level() === 0) { ob_start(); }

// Bootstrap principal (traz $connect, $urlPath, $translation, $settings, etc.)
require __DIR__ . '/core.php';

/**
 * Decide o redirecionamento pós-login:
 * - Se o usuário for anunciante (tem registro em sellers), vai para o dashboard
 * - Caso contrário, vai para a home pública.
 */
function nd_redirect_after_login($connect, $urlPath, ?array $userRow = null): void {
    // Garante sessão
    if (session_status() !== PHP_SESSION_ACTIVE) {
        @session_start();
    }

    $isSeller  = false;
    $sellerId  = null;
    $userRow   = $userRow ?? null;

    // Se não recebemos o row completo, tenta buscar pelo e-mail da sessão
    if ($userRow === null && !empty($_SESSION['user_email'] ?? '')) {
        $email = (string)$_SESSION['user_email'];
        try {
            if ($connect instanceof PDO) {
                $stmtUser = $connect->prepare("SELECT * FROM users WHERE user_email = :email AND user_status = 1 LIMIT 1");
                $stmtUser->execute([':email' => $email]);
                $userRow = $stmtUser->fetch(PDO::FETCH_ASSOC) ?: null;
            }
        } catch (Exception $e) {
            $userRow = null;
        }
    }

    if (is_array($userRow) && !empty($userRow)) {
        $userId = (int)($userRow['user_id'] ?? 0);
        if ($userId > 0 && $connect instanceof PDO) {
            try {
                $stmtSeller = $connect->prepare("SELECT seller_id FROM sellers WHERE seller_user = :uid AND seller_status = 1 LIMIT 1");
                $stmtSeller->execute([':uid' => $userId]);
                $sellerRow = $stmtSeller->fetch(PDO::FETCH_ASSOC);
                if ($sellerRow && isset($sellerRow['seller_id'])) {
                    $isSeller = true;
                    $sellerId = (int)$sellerRow['seller_id'];
                    $_SESSION['seller_id'] = $sellerId;
                }
            } catch (Exception $e) {
                // silencia falha de detecção; segue fluxo padrão
            }
        }
    }

    if ($isSeller) {
        nd_safe_redirect('/dashboard/home.php');
    } else {
        nd_safe_redirect($urlPath->home());
    }
}


// Redirect helper (sem enviar saída)
if (!function_exists('nd_safe_redirect')) {
    function nd_safe_redirect(string $url, int $code = 302): void {
        while (function_exists('ob_get_length') && ob_get_length()) { @ob_end_clean(); }
        if (!headers_sent()) {
            header('Location: ' . $url, true, $code);
            exit;
        }
        echo '<!doctype html><meta http-equiv="refresh" content="0;url=' . htmlspecialchars($url, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8') . '">';
        exit;
    }
}

// =====================
// SEO (i18n com fallback)
// =====================
$signinLabel      = $translation['tr_signinpage'] ?? ($translation['LOGIN'] ?? 'Login');
$seoBaseTitle     = $translation['tr_1']          ?? '';
$seoBaseDesc      = $translation['tr_3']          ?? '';

$titleSeoHeader       = getSeoTitle($seoBaseTitle, $signinLabel);
$descriptionSeoHeader = getSeoDescription($seoBaseDesc);

// =====================
// Validação de formulário
// =====================
$errors = [];
$validateEmail = false;
$validatePassword = false;

if (isLogged()) {
    nd_redirect_after_login($connect, $urlPath, null);
}

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'POST') {
    $user_email = htmlspecialchars(strtolower($_POST['user_email'] ?? ''), ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    $user_password = (string)($_POST['user_password'] ?? '');
    $encrtypted_password = hash('sha512', $user_password);

    if ($user_email === '') {
        $errors[] = $translation['tr_158'] ?? 'Por favor, informe o e-mail.';
    } elseif (!filter_var($user_email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = $translation['tr_163'] ?? 'E-mail inválido.';
    } else {
        $validateEmail = true;
    }

    if ($user_password === '') {
        $errors[] = $translation['tr_160'] ?? 'Por favor, informe a senha.';
    } else {
        $validatePassword = true;
    }

    if ($validatePassword && $validateEmail) {
        try {
            // $connect vem do core.php
            $connect->query('SELECT 1'); // ténue ping
        } catch (PDOException $e) {
            $errors[] = $translation['tr_168'] ?? 'Erro de conexão.';
        }

        // Existe e está ativo?
        if (empty($errors)) {
            $statement = $connect->prepare("SELECT * FROM users WHERE user_email = :user_email AND user_status = 1");
            $statement->execute([':user_email' => $user_email]);
            $result = $statement->fetch();

            if ($result == false) {
                $errors[] = $translation['tr_172'] ?? 'Usuário não encontrado ou inativo.';
            }

            // Se o site exigir verificados
            if (($settings['st_login_verified_users_only'] ?? 0) == 1) {
                $statement = $connect->prepare("SELECT * FROM users WHERE user_email = :user_email AND user_verified = 1");
                $statement->execute([':user_email' => $user_email]);
                $result = $statement->fetch();

                if ($result == false) {
                    $errors[] = $translation['tr_352'] ?? 'Conta não verificada.';
                }
            }
        }

        if (empty($errors)) {
            $statement = $connect->prepare("SELECT * FROM users WHERE user_email = :user_email AND user_password = :user_password AND user_status = 1");
            $statement->execute([
                ':user_email' => $user_email,
                ':user_password' => $encrtypted_password
            ]);

            $result_login = $statement->fetch();

            if ($result_login !== false) {
                $_SESSION['signedin']   = true;
                $_SESSION['user_email'] = $user_email;
                $_SESSION['user_name']  = $result_login['user_name'] ?? '';

                nd_redirect_after_login($connect, $urlPath, $result_login);
            } else {
                $errors[] = $translation['tr_174'] ?? 'E-mail ou senha incorretos.';
            }
        }
    }
}

// =====================
// Renderização
// =====================

// Header (inclui o seletor padronizado via partial dentro do header.php)
require __DIR__ . '/header.php';

// View principal da página de login
require __DIR__ . '/views/signin.view.php';

// Footer
require __DIR__ . '/footer.php';

// Dicas:
// - Não inserir HTML antes da chamada do header.php
// - Não reintroduzir scripts duplicados do Google Translate aqui.
