<?php
declare(strict_types=1);

/**
 * Hardened logout (signout) endpoint
 * - Avoids 'headers already sent'
 * - Properly clears session + cookie
 * - Prevents cache
 * - Compatible with existing $urlPath->home() routing
 */

// Avoid "headers already sent" noise if any included file prints accidentally
if (!headers_sent()) { @ob_start(); }

require __DIR__ . '/core.php'; // keeps $urlPath, isLogged(), etc.

// Start/ensure session so we can clear it safely
if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }

// If you store user email in cookie for 2FA trust, clear it (best-effort)
$twofa_lib = __DIR__ . '/admin/twofa_lib.php';
if (is_file($twofa_lib)) {
    require_once $twofa_lib;
    if (function_exists('twofa_cookie_clear') && !empty($_SESSION['user_email'])) {
        try { twofa_cookie_clear($_SESSION['user_email']); } catch (\Throwable $e) {}
    }
}

// Clear all session variables
$_SESSION = [];

// Delete the session cookie (if any)
if (ini_get('session.use_cookies')) {
    $params = session_get_cookie_params();
    setcookie(session_name(), '', time() - 42000,
        $params['path'] ?? '/',
        $params['domain'] ?? '',
        (bool)($params['secure'] ?? false),
        (bool)($params['httponly'] ?? true)
    );
}

// Destroy the session storage
@session_destroy();

// Prevent browser caching of this response
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

// ==== patched: compute destination safely (absolute vs relative) ====
$home = '';
try {
    if (isset($urlPath) && method_exists($urlPath, 'home')) {
        $home = (string) $urlPath->home();
    } elseif (defined('SITE_URL')) {
        $home = (string) SITE_URL;
    }
} catch (\Throwable $e) {
    // ignore, fallback below
}
if ($home === '') { $home = '/'; }

$is_abs = (bool) preg_match('~^https?://~i', $home);
if (!$is_abs) {
    if ($home[0] !== '/') { $home = '/' . ltrim($home, '/'); }
}
$dest = $home . (strpos($home, '?') === false ? '?logout=1' : '&logout=1');
// ==== /patched ====

header('Location: ' . $dest, true, 303);
exit;
